/*
 * Copyright (c) 2004 Jean-Franois Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib.h>
#include <gmodule.h>

#include "printing-system.h"

/* Prototypes */
static gint compare_name_func (const Printer *pr, const gchar *name);
static gint compare_alias_func (const Printer *pr, const gchar *alias);

GList* (*printing_system_get_printers_list) (void) = NULL;
void (*printing_system_main_window_customize) (ManagerDlg *dlg) = NULL;
gboolean (*printing_system_set_default_printer) (const gchar *printer) = NULL;
Printer* (*printing_system_get_default_printer) (GList *printers) = NULL;

void (*printing_system_queue_dialog_customize) (QueueDlg *dialog) = NULL;
gboolean (*printing_system_remove_job) (const gchar *printer, gint id) = NULL;
GList* (*printing_system_get_jobs_list) (const gchar *printer) = NULL;
void (*printing_system_info_init) (PrintingSystemInfo * psinfo) = NULL;

void (*printing_system_properties_dialog_run) (GtkWidget *widget, const gchar *printer) = NULL;

void (*printing_system_about_dialog_run) (GtkWidget *widget, gpointer data) = NULL;

void (*printing_system_print_dialog_customize) (PrintDlg *dialog) = NULL;
gboolean (*printing_system_print_file) (PrintDlg *dialog, const gchar *file) = NULL;

/*************/
/* Functions */
/*************/
void printing_system_info_free (PrintingSystemInfo *psinfo)
{
  g_free (psinfo->name);
  g_free (psinfo->description);
  g_free (psinfo->version);
  g_free (psinfo->author);
  g_free (psinfo->homepage);

  g_free (psinfo);
}

void printer_free (Printer *printer, gpointer user_data)
{
  g_free (printer->name);
  g_free (printer->alias);
}

void job_free (Job *job, gpointer user_data)
{
  g_free (job->name);
  g_free (job->user);
  g_free (job->state);
  g_free (job->size);
  g_free (job->priority);
  g_free (job->creation_time);
  g_free (job->processing_time);
}

Printer* printer_lookup_byalias (GList *list, const gchar *alias)
{
  GList *el;

  el = g_list_find_custom (list, alias, (GCompareFunc)compare_alias_func); 

  if(el)
    return ((Printer *) el->data);
  else
    return NULL;
}

Printer* printer_lookup_byname (GList *list, const gchar *name)
{
  GList *el;

  el = g_list_find_custom (list, name, (GCompareFunc)compare_name_func); 

  if(el)
    return ((Printer *) el->data);
  else
    return NULL;
}

gboolean load_printing_system (const gchar *path)
{
  GModule *gm;
  gm = g_module_open (path, 0);
  
  if (!gm)
    return FALSE;

  if(!g_module_symbol (gm, "printing_system_info_init_impl",
		       (gpointer) &printing_system_info_init)){
    g_module_close (gm);
    g_warning ("printing_system_info_init_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_get_printers_list_impl",
		       (gpointer) &printing_system_get_printers_list)){
    g_module_close (gm);
    g_warning ("printing_system_get_printers_list_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_main_window_customize_impl",
		       (gpointer) &printing_system_main_window_customize)){
    g_module_close (gm);
    g_warning ("printing_system_main_window_customize_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_remove_job_impl",
		       (gpointer) &printing_system_remove_job)){
    g_module_close (gm);
    g_warning ("printing_system_remove_job_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_get_jobs_list_impl",
		       (gpointer) &printing_system_get_jobs_list)){
    g_module_close (gm);
    g_warning ("printing_system_get_jobs_list_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_queue_dialog_customize_impl",
		       (gpointer) &printing_system_queue_dialog_customize)){
    g_module_close (gm);
    g_warning ("printing_system_queue_dialog_customize_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_properties_dialog_run_impl",
		       (gpointer) &printing_system_properties_dialog_run)){
    g_module_close (gm);
    g_warning ("printing_system_properties_dialog_run_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_set_default_printer_impl",
		       (gpointer) &printing_system_set_default_printer)){
    g_module_close (gm);
    g_warning ("printing_system_set_default_printer_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_get_default_printer_impl",
		       (gpointer) &printing_system_get_default_printer)){
    g_module_close (gm);
    g_warning ("printing_system_get_default_printer_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_about_dialog_run_impl",
		       (gpointer) &printing_system_about_dialog_run)){
    g_module_close (gm);
    g_warning ("printing_system_about_dialog_run_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_print_dialog_customize_impl",
		       (gpointer) &printing_system_print_dialog_customize)){
    g_module_close (gm);
    g_warning ("printing_system_print_dialog_customize_impl symbol not found");

    return FALSE;
  }
  if(!g_module_symbol (gm, "printing_system_print_file_impl",
		       (gpointer) &printing_system_print_file)){
    g_module_close (gm);
    g_warning ("printing_system_print_file_impl symbol not found");

    return FALSE;
  }

  return TRUE;
}

/*************/
/* Internals */
/*************/
static gint compare_name_func (const Printer *pr, const gchar *name)
{
  return (g_ascii_strcasecmp (pr->name, name));
}

static gint compare_alias_func (const Printer *pr, const gchar *alias)
{
  return (g_ascii_strcasecmp (pr->alias, alias));
}

