/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <errno.h>
#include <limits.h>
#include <memory.h>
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#elif HAVE_VARARGS_H
#include <varargs.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <dbh.h>

#ifdef XFCE_DISABLE_DEPRECATED
#undef XFCE_DISABLE_DEPRECATED
#endif
#include <libxfcegui4/dialogs.h>
#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "modules.h"

#include "run.h"

#include "terminal.h"
#include "input.h"
#include "misc.h"
#include "entry.h"
#include "xmcs-manager.h"


static 
void 
save_to_open_history(		gchar *path2save, 
				gchar *in_cmd, 
				int in_term)
{
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *dbh_file=g_build_filename(xdg_dir,OPEN_FILE,NULL);
    GString *gs;
    DBHashTable *d;
    gchar *data;
    int length;
    gchar *basename;

    g_free(xdg_dir);

    if (!path2save) return;
    basename=g_path_get_basename(path2save);
    if ((d=DBH_open(dbh_file))==NULL){
    	if ((d=DBH_create(dbh_file, 11))==NULL) {
		unlink(dbh_file);
		if ((d=DBH_create(dbh_file, 11))==NULL)	return;
	}
    }
    gs = g_string_new(basename);
    g_free(basename);
    sprintf((char *)DBH_KEY(d), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    data = (gchar *)DBH_DATA(d);
    if (in_term) *data = 1;
    else *data=0;
    data++;
    
    length=strlen(in_cmd)+2;
    if (length > DBH_MAXIMUM_RECORD_SIZE(d))length = DBH_MAXIMUM_RECORD_SIZE(d);
    
    strncpy(data,in_cmd, length-2);
    DBH_set_recordsize (d,length);
    
    DBH_update(d);
    DBH_close(d);
    g_free(dbh_file);
}

G_MODULE_EXPORT
const 
gchar *
get_from_open_history(		gchar *path2find, 
				int *in_term){
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *dbh_file=g_build_filename(xdg_dir,OPEN_FILE,NULL);
    GString *gs;
    DBHashTable *d;
    gchar *data;
    gchar *basename=g_path_get_basename(path2find);
    static gchar *in_cmd=NULL;

    g_free(xdg_dir);
    
    if ((d=DBH_open(dbh_file))==NULL){
    	if ((d=DBH_create(dbh_file, 11))==NULL) {
		unlink(dbh_file);
		if ((d=DBH_create(dbh_file, 11))==NULL)	{
		    g_free(basename);
		    g_free(dbh_file);
		    return NULL;
		}
	}
    }
    g_free(dbh_file);
    gs = g_string_new(basename);
    g_free(basename);
    sprintf((char *)DBH_KEY(d), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    data = (gchar *)DBH_DATA(d);

    if (!DBH_load(d)) return NULL;
    

    
    if (*data==0) *in_term = 0;
    else *in_term=1;
    data++;
    g_free(in_cmd);
    in_cmd=g_strdup(data);
    
    DBH_close(d);
    return (const gchar *)in_cmd;
}


G_MODULE_EXPORT
int 
runvwd(			const gchar *workdir,
			char *argv[])
{

    int i;
    GError *error=NULL;
    
    if (!argv || !argv[0]) return FALSE;
    if (!workdir || !g_file_test(workdir,G_FILE_TEST_IS_DIR)) {
	workdir=get_selected_chdir();
	if (!g_file_test(workdir,G_FILE_TEST_IS_DIR)) workdir=GETWD;
    }
    

    if (strcmp(workdir,GETWD)!=0) {
	print_diagnostics(NULL,"$chdir ",workdir,"\n$",argv[0],NULL);
	xffm_setenv("PWD",(gchar *)workdir,FALSE);
#ifdef DEBUG
	printf("DBG: getenv(PWD)=%s\n",getenv("PWD"));
#endif
    }
    else {
	print_diagnostics(NULL,"$",argv[0],NULL);
    }
    for (i=1;argv[i];i++) print_diagnostics(NULL," ",argv[i],NULL);
    print_diagnostics(NULL,"\n",NULL);
    print_status("xfce/info",_("Executing")," ",argv[0],NULL);

    if (strcmp(argv[0],"sudo")==0) print_status("xfce/sudo", _("Executing"), " sudo ", argv[1], NULL);
    else print_status("xfce/info", _("Executing"), " ", argv[0], NULL);

    if (!g_spawn_async(workdir,argv,NULL,G_SPAWN_SEARCH_PATH,NULL,NULL,NULL,&error)){
	  gchar *msg = g_strcompress (error->message);
	  print_diagnostics("xfce/error",msg,":\n",argv[0] ,"\n",NULL);
	  g_error_free (error);
	  g_free (msg);
	  chdir (GETWD);
	  xffm_setenv("PWD",(gchar *)GETWD,FALSE);
	  return FALSE;
    }

    if (strcmp(workdir,GETWD)!=0) print_diagnostics(NULL,"$chdir ",GETWD,"\n",NULL);
    
    xffm_setenv("PWD",(gchar *)GETWD,FALSE);

    chdir(GETWD);
    return TRUE;

}


G_MODULE_EXPORT
void 
save_flags(		gchar *in_cmd, 
			gboolean interm, 
			gboolean hold)
{
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *g=g_build_filename(xdg_dir,RUN_FLAG_FILE,NULL);

    g_free(xdg_dir);
    
    if((runflags = DBH_open(g)) == NULL)
    {
	if((runflags = DBH_create(g, 11)) == NULL){
	    g_warning("Cannot create %s\n",g);
	    return;
	}
    }
    gs = g_string_new(in_cmd);
    sprintf((char *)DBH_KEY(runflags), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    /*printf("DBG: bookmarking with key=%s\n",d_path);*/
    flags = (int *)runflags->data;
    flags[0]=interm;
    flags[1]=hold;
    DBH_set_recordsize(runflags, 2*sizeof(int));
    
    DBH_update(runflags);
    DBH_close(runflags);
    D(printf("flags saved in dbh file for %s\n",in_cmd);)
}

G_MODULE_EXPORT
void
recover_flags(		gchar *in_cmd,
			gboolean *interm,
			gboolean *hold)
{
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    gchar *g=g_build_filename(xdg_dir,RUN_FLAG_FILE,NULL);

    g_free(xdg_dir);
    
    if((runflags = DBH_open(g)) == NULL)
    {
	    D(g_warning("Cannot open %s\n",g);)
	    *interm=0;
	    *hold=0;
	    return;
    }
    gs = g_string_new(in_cmd);
    sprintf((char *)DBH_KEY(runflags), "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);
    /*printf("DBG: bookmarking with key=%s\n",d_path);*/
    flags = (int *)runflags->data;
    DBH_load(runflags);
    *interm = flags[0];
    *hold = flags[1];
    DBH_close(runflags);

    D(printf("flags recovered from dbh file for %s, interm=%d hold=%d\n",in_cmd,*interm,*hold);)
}

/* this function is taken and modified from libxfce4gui (see credits in dialogs.c): */
static 
gboolean
xfce_confirm_custom (		const gchar *text,
				const gchar *action_false,
				const gchar *action_true)
{
    GtkWidget *dialog, *button;
    int response = GTK_RESPONSE_NONE;

    if (!action_false || !action_true) {
	g_error("!action_false || !action_true");
    }

    dialog = gtk_message_dialog_new (NULL,
				     GTK_DIALOG_MODAL,
				     GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
				     text);

    button = mixed_button_new ("gtk-no",action_false);    
    gtk_widget_show (button);
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_NO);

    button = mixed_button_new ("gtk-yes",action_true);
    gtk_widget_show (button);
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);

    /*gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);*/
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);

    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);
    gtk_widget_destroy (dialog);

    if (response == GTK_RESPONSE_YES) return TRUE;
    else return FALSE;
}


G_MODULE_EXPORT
int 
on_run_path(		char *in_cmd, 
			char *path, 
			gboolean interm, 
			gboolean remember, 
			gboolean put_in_history,
			gboolean hold)
{
    GError *error=NULL;
    const gchar *command;
    gchar *name;
    int argc;
    gchar **argv;
    gchar *wd=NULL;
    if (!g_file_test(in_cmd,G_FILE_TEST_EXISTS) && !MIME_is_valid_command(in_cmd)){
	print_status("xfce/error",strerror(ENOEXEC),": ",in_cmd,NULL);
	return FALSE;
    }

    if(getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM"))) hold=TRUE;
    
    /* why did I ever use a relative path here? Reverting cause it breaks
     * doubleclick on .html files...
     * if (path) name=g_path_get_basename(path); else name=g_strdup("");
     * */
    if (path) name=g_strdup(path); else name=g_strdup("");
    
    command = MIME_mk_command_line(in_cmd, name, interm, hold);
    if (!command){
    	if (name) g_free(name);   
	return FALSE;
    }
    if (name) g_free(name); 
    name = NULL; 

    if (path) {
      wd=g_path_get_dirname((const gchar *)path);	    
    } else if (g_file_test(in_cmd,G_FILE_TEST_EXISTS)){
      wd=g_path_get_dirname((const gchar *)in_cmd);
    } 
    
#ifdef DEBUG
    printf("in_cmd=%s\n",command);
#endif
    g_shell_parse_argv (command, &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics("xfce/error",msg,":\n",command,"\n",NULL);
	g_error_free(error);
	g_free (msg);
	g_free (wd);
	return FALSE;
    }
    if (runvwd(wd,argv)){
 	save_to_open_history(path,in_cmd,interm);
    }
    g_free(wd); 
    g_strfreev (argv);
    
    if (put_in_history){
      gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
      gchar *f=g_build_filename(xdg_dir,RUN_DBH_FILE,NULL);

      g_free(xdg_dir);
      XFC_save_to_history(f,in_cmd);
      save_flags(in_cmd,interm,hold);
      g_free(f);
    }
    
    if(path && remember)
    {
#if 10
	/* multiple dots? */
	gchar *p=g_path_get_basename(path);
	gchar *foo=NULL;
	if (p && strchr(p,'.') && strchr(p,'.') != strrchr(p,'.') ){
	    gchar *m=g_strdup_printf(_("Remember %s or %s?"),strchr(p,'.'),strrchr(p,'.'));
	    if (xfce_confirm_custom ((const gchar *)m,
				     (const gchar *)strchr(p,'.'),
				     (const gchar *)strrchr(p,'.'))){
		foo=g_strdup_printf("foo%s",strrchr(p,'.'));
	    } else {
		foo=g_strdup_printf("foo%s",strchr(p,'.'));
	    }	
	} else {
		foo=g_strdup(path);
	}
#endif
	if(strlen(path))
	{
	   command = MIME_mk_command_line(in_cmd, NULL, interm, hold);
	
#ifdef DEBUG
	    printf("DBG: adding %s --> %s\n",path,in_cmd); 
#endif
	    print_diagnostics("xfce/info",foo,"-->",command,"\n",NULL);
	    MIME_add(foo,command);
	}
	g_free(p);
	g_free(foo);
    }
    return TRUE;
}

G_MODULE_EXPORT
int 
on_run(		char *in_cmd, 
		tree_entry_t * en, 
		gboolean in_terminal, 
		gboolean remember, 
		gboolean put_in_history, 
		gboolean hold)
{
	char *path=NULL;
	if (en && en->path) path=en->path;
	return on_run_path(in_cmd, path, in_terminal, remember, put_in_history, hold);
}




G_MODULE_EXPORT
void 
set_run_combo(		xfc_combo_info_t *combo_info)
{
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    GtkTreeIter iter;
    tree_entry_t *en;
    gchar *f=g_build_filename(xdg_dir,RUN_DBH_FILE,NULL); 
    gboolean update=FALSE;
    gboolean docheck=TRUE;

    if (access(f,F_OK)!=0){update=TRUE; docheck=FALSE;}
    else {
       struct stat st[2];	    
       if (stat(f,st) < 0) update=TRUE;
       else {
	 gchar *m=g_strconcat(APPLICATION_MIME_FILE,NULL);
	 if (stat(m,st+1) >= 0 && st[0].st_mtime < st[1].st_mtime) update=TRUE;
	 m=g_strconcat(xdg_dir,LOCAL_APPLICATION_MIME_FILE,NULL);
	 if (stat(m,st+1) >= 0 && st[0].st_mtime < st[1].st_mtime) update=TRUE;
       }
    }
    
    XFC_read_history(combo_info,f);

    en = get_selected_entry(&iter);
    if (en) {
	const gchar *p=MIME_command(en->path);
    	if (p) combo_info->list = g_list_prepend(combo_info->list,g_strdup(p) );
    }
    XFC_set_combo(combo_info,NULL);
    g_free(f);

    {
      gboolean interm,hold;
      GtkEntry * entry = (GtkEntry *) WIDGET("input_entry");
      gchar *choice = g_strdup((gchar *)gtk_entry_get_text(entry));
      GtkWidget *check = WIDGET("checkbutton1");
      recover_flags(choice,&interm, &hold);
      gtk_toggle_button_set_active ((GtkToggleButton *)check,interm);
      check = WIDGET("checkbutton3");
      gtk_toggle_button_set_active ((GtkToggleButton *)check,hold);
      if(getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM"))) {
         gtk_toggle_button_set_active ((GtkToggleButton *)check,TRUE);
	 gtk_widget_set_sensitive(check, FALSE);
      } else {
	 gtk_widget_set_sensitive(check, TRUE);
      }
      
      if (!interm) gtk_widget_hide(check); else gtk_widget_show(check); 
      g_free(choice);
    }
    g_free(xdg_dir);

    return;
}

static gchar *script_types[]={
    "application/x-csh",
    "application/x-bsh",
    "application/x-ksh",
    "application/x-shellscript",
    "application/x-tsh",
    "application/x-zsh",
    "application/x-perl",
    NULL
};

static 
gboolean 
is_a_script(const gchar *path)
{ 
  const gchar *t=MIME_get_type((const gchar *)path, TRUE);
  if (t) {
      int i;
      for (i=0;script_types[i];i++) if (strncmp(script_types[i],t,strlen(script_types[i]))==0){
	  return TRUE;
      }
  }
  return FALSE;
}

G_MODULE_EXPORT
void 
double_click_run(	tree_entry_t *en)
{
    GError *error=NULL;
    char *a=NULL;
    int argc;
    gchar **argv;
    gboolean interm=FALSE;
    /* interm specified? */
    /* obsolete (from xfapps branch):
     * if (IS_IN_TERM(en->subtype)) interm=TRUE; */
 
    if (!IS_EXE(en->type)) return;
    
    /* is it a script? */
    if (is_a_script(en->path)) interm=TRUE;
    
    if (interm){
	a = g_strdup_printf("%s -e \"%s\"",what_term(),en->path);
    } else {
        a=g_strdup_printf("\"%s\"",en->path);
    }
    
    g_shell_parse_argv (a, &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics("xfce/error",msg,":\n",a,"\n",NULL);
	g_error_free(error);
	g_free(a);
	g_free (msg);
	return;
    }
    runv(argv);
    g_strfreev (argv); 
    g_free(a);
    return;
}


G_MODULE_EXPORT
void 
double_click_open_with(		tree_entry_t *en)
{
    /* open with */
    gchar *name;
    int argc;
    gchar **argv;
    const gchar *command;
    const gchar *command_fmt;
    GError *error=NULL;
    
    if (!en || !en->path) return;
    name=g_path_get_basename(en->path);
    
    
    command_fmt = MIME_command(name);
    
    if(!command_fmt)
    {
	show_input(RUN_DOUBLE_CLICK);
	print_status("xfce/question",name,NULL);
    } else {
        gchar *wd=g_path_get_dirname((const gchar *)en->path);	    
	command =  MIME_mk_command_line(command_fmt,name,FALSE,FALSE);


	g_shell_parse_argv (command, &argc,&argv,&error);
	if (error){
	    gchar *msg = g_strcompress (error->message);
	    print_diagnostics("xfce/error",msg,":\n",command,"\n",NULL);
	    g_error_free(error);
	    g_free (msg);
	    g_free(wd);
	    return;
	}
	runvwd(wd,argv);
	g_free(wd);
	g_strfreev (argv); 
    }
    g_free(name);
    return;
}



/* callbacks */
G_MODULE_EXPORT
void tb_run(GtkButton * button, gpointer user_data)
{
    show_input(RUN_INPUT);
}

G_MODULE_EXPORT
void sb_run(GtkWidget *w){
    show_input(RUN_INPUT);
}

G_MODULE_EXPORT
void on_open_with_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    tree_entry_t *en;
    GtkTreeIter iter;
    en = get_selected_entry(&iter);
    if(!en)
	show_input(RUN_INPUT);
    else
    {
	show_input(RUN_DOUBLE_CLICK);
	print_status("xfce/question", _("Input requested"), NULL);
    }


}

G_MODULE_EXPORT
void on_run_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    show_input(RUN_INPUT);
}
