/*
 *  icontheme -  lookup themed icons.
 *
 *  Copyright (c) 2004 Edscott Wilson Garcia <edscott@imp.mx>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*#define DEBUG*/
#ifdef DEBUG
#define D(x) x
#else
#define D(x) 
#endif

#define DD(x) 

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <regex.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtksettings.h>
#include <gmodule.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>


#define DEFAULT_THEME "Rodent"
#define FALLBACK "hicolor" 

#define VALID_KDEDIR g_getenv("KDEDIR") && strlen(g_getenv("KDEDIR"))
#define KDEDIR g_getenv("KDEDIR"),"share","icons"
#define HOMEDIR xfce_get_homedir(), ".icons"
#ifndef DATADIR
#define DATADIR NULL
#endif

static GList *base_dirs=NULL;
static GList *theme_list=NULL;

typedef struct icon_info_t {
    gchar *path;
    int size;  
    gchar *type;
    struct icon_info_t *next;
}icon_info_t;

typedef struct smallhash_t {
    gchar *context;
    GHashTable *hash;
}smallhash_t;

typedef struct theme_info_t {
    gchar *id;  /* untranslated theme name */
    GHashTable *bighash; /* hash of icon_info_t */
    GList *smallhashes;  /* context hashes to speed access to bighash */ 
}theme_info_t;


static
const gchar *
get_supported_regex(void){
    GSList *pix_formats, *l;
    gchar **pix_extensions,**p; 
    static gchar *reg=NULL,*r=NULL;
    /* check SVG format support */
    if ((pix_formats = gdk_pixbuf_get_formats()) != NULL) {
	for(l = pix_formats; l; l = l->next) {
	    GdkPixbufFormat *fmt = l->data;
	    pix_extensions = gdk_pixbuf_format_get_extensions(fmt);
	    for(p=pix_extensions; *p; p++) {
		D(printf("supported=%s\n",*p);)
		if (reg) {
		    g_free(r);
		    r=reg;
		    reg=g_strconcat(r,"|",*p,NULL);
		} else reg=g_strdup(*p);
	    }
	    g_strfreev(pix_extensions);
	}
	g_slist_free(pix_formats);
    }
    if (!reg) return "\\.(png|xpm)$)";
    g_free(r);
    r=g_strconcat("\\.(",reg,")$",NULL);
    g_free(reg); reg=NULL;

    D(printf("regex=%s\n",r);)
    return (const gchar *) r;
}
    

static
gint  
compare_theme_info(gconstpointer a,gconstpointer b){
    theme_info_t *theme_info_a = (theme_info_t *)a;
    theme_info_t *theme_info_b = (theme_info_t *)b;
    return strcmp(theme_info_a->id, theme_info_b->id);
}


static 
GList * 
free_string_list (GList *list){	
    if (list){
	GList *tmp;
	for (tmp=list; tmp; tmp = tmp->next) g_free(tmp->data);
	g_list_free(list);
    }
    return NULL;
}



static 
void 
clear_bighash(gpointer key, gpointer value, gpointer user_data)
{
    icon_info_t *p,*icon_info_p = (icon_info_t *)value;
    g_free(key);
    if (!value) return;
    do {
	g_free(icon_info_p->path);
	g_free(icon_info_p->type);
	p=icon_info_p;
	icon_info_p = icon_info_p->next;
	g_free(p);
    } while (icon_info_p);
    return;
}

    
/*frees Glist of themes */
static 
GList * 
free_theme_list (GList *list){
    GList *tmp=list, *c_list;
    if (!list) return NULL;
    for (tmp=list; tmp; tmp = tmp->next) {
	    theme_info_t *theme_info_p;
	    theme_info_p= (theme_info_t *)tmp->data;
	    /* smallhashes has no allocated memory in hash table. All pointers
	     * belong to the bighash. */
	    for (c_list=theme_info_p->smallhashes; c_list; c_list=c_list->next){
		smallhash_t *smallhash = (smallhash_t *)c_list->data;
		g_free(smallhash->context);
		/* pointers within small hashes are duplicated in bighash,
		 * so that they need not be freed at this step */
		g_hash_table_destroy(smallhash->hash);
		g_free(smallhash);
	    }
	    /* now we free memory */
	    g_list_free(theme_info_p->smallhashes);
	    g_hash_table_foreach(theme_info_p->bighash, clear_bighash, NULL);
            g_hash_table_destroy(theme_info_p->bighash);	    
	    g_free(theme_info_p->id);
	    g_free(theme_info_p);
    }
    return NULL;
}

static
gchar *
dump_if_duplicate(gchar *X,gchar *Y){
    if (Y) {
	if (strcmp(X,Y)==0) {
	    g_free(Y);
	    return NULL;
	}
    }
    return Y;
}

#ifdef DEBUG
static
void 
showme_themeinfo(GList *list, gchar *tag){
    {
	GList *tmp=list;
	printf("%s:\n",tag);
	for (; tmp; tmp=tmp->next){
	    theme_info_t *theme_info = (theme_info_t *)tmp->data;
	    if (!theme_info) continue;
	    printf("\t%s\n",(gchar *)theme_info->id);
	}
    }
}
static
void 
showme(GList *list, gchar *tag){
    {
	GList *tmp=list;
	printf("%s:\n",tag);
	for (; tmp; tmp=tmp->next)
	    printf("\t%s\n",(gchar *)tmp->data);
    }
}

#endif

static 
void 
chop (gchar *g){
    if (!g || strlen(g)<2) return;
    if (*(g+(strlen(g)-1)) == G_DIR_SEPARATOR) *(g+(strlen(g)-1)) = 0;
}

static
GList *
get_base_dirs(GList *base_dirs){
    gchar *kdedir=NULL, *datadir=NULL, *homedir=NULL;
    gchar **p, **paths;
    
    base_dirs = free_string_list (base_dirs);
    
    if (VALID_KDEDIR) kdedir=g_build_filename(KDEDIR,NULL);
    if (DATADIR) datadir=g_build_filename(DATADIR, "icons", NULL);
    homedir=g_build_filename(HOMEDIR, NULL);
    
    /* XFCE_RESOURCE_DATA, "icons/ */
    paths = xfce_resource_lookup_all(XFCE_RESOURCE_DATA, "icons/");
    D(if (!paths) printf("DBG: paths is NULL\n");)
    for (p=paths; *p; p++) {
	chop(*p);
	D(printf("DBG: path:%s \n",*p);)
	if (g_file_test(*p,G_FILE_TEST_IS_DIR))
	    base_dirs=g_list_append(base_dirs,(gpointer)g_strdup(*p));
	D(else g_warning("!G_FILE_TEST_EXISTS");)
	datadir=dump_if_duplicate(*p,datadir);
	kdedir=dump_if_duplicate(*p,kdedir);
	homedir=dump_if_duplicate(*p,homedir);
    }
    g_strfreev(paths); 
    /* KDE path and datadir */
    if (kdedir) base_dirs=g_list_append(base_dirs,(gpointer)kdedir);
    if (datadir) base_dirs=g_list_append(base_dirs,(gpointer)datadir);
    if (homedir) base_dirs=g_list_prepend(base_dirs,(gpointer)homedir);
#ifdef DEBUG
    showme(base_dirs,"base_dirs");
#endif
    return base_dirs;
}

static
gchar ** 
get_rc_info(gchar *index,gchar *rc_info){
    XfceRc *themerc;
    gchar **info=NULL;
    if((themerc = xfce_rc_simple_open(index, TRUE)) == NULL) return NULL;
    if(xfce_rc_has_group(themerc, "Icon Theme")) {
        xfce_rc_set_group(themerc, "Icon Theme");
        info = xfce_rc_read_list_entry(themerc, rc_info, ",");
    }
    xfce_rc_close(themerc);
    return info;
}

static
GList *
add_theme_to_list(GList *list, const gchar *id){
    theme_info_t *theme_info_p=(theme_info_t *)malloc(sizeof(theme_info_t));
    theme_info_p->id = g_strdup(id);
    theme_info_p->smallhashes = NULL;
    theme_info_p->bighash = g_hash_table_new(g_str_hash, g_str_equal);
    list = g_list_append(list,(gpointer)theme_info_p);
    return list;
}

static 
gchar *
theme_index(gchar *base_dir, gchar *theme){
    gchar *index;
    chop(base_dir);
    index=g_build_filename(base_dir, theme, NULL);
    if (!g_file_test(index,G_FILE_TEST_IS_DIR)){
        g_free(index);
        return NULL;
    }
    g_free(index);
    index=g_build_filename(base_dir, theme, "index.theme", NULL);
    if (!g_file_test(index,G_FILE_TEST_EXISTS)){
        g_warning("%s does not exist",index);
        g_free(index);
        return NULL;
    }
    return index;
}

static
GList *
add_theme_name(GList *list, gchar *base_dir, gchar *theme){
    gchar *index=theme_index(base_dir, theme);
    if (!index) return list;
    else {
	theme_info_t theme_info;
	theme_info.id=theme;
	if (!g_list_find_custom (theme_list,&theme_info, compare_theme_info)){
	    list = add_theme_to_list(list,theme);
	}
	D(g_message("using theme defined with %s",index);)
	g_free(index);
	return list;
    }
}

static
gboolean 
read_icon_directory(gchar *directory, theme_info_t *theme_info_p, 
		int size, const gchar *type, const gchar *context){
    GDir *dir;
    icon_info_t *icon_info_p;
    static regex_t supported;
    static gboolean regex_compiled=FALSE;
    const gchar *name;
    gchar *key;
    
    if (!regex_compiled) {
	const gchar *regex=get_supported_regex();
	if (regcomp(&supported, regex, REG_EXTENDED | REG_ICASE | REG_NOSUB)==0) regex_compiled=TRUE;
	else regex_compiled=FALSE;
    }
    dir = g_dir_open(directory, 0, NULL);
    D(printf("DBG: reading %s...(%s)\n",directory,((dir)?"yes":"no"));)
    if (!dir) return FALSE;
     
    while((name = g_dir_read_name(dir))) {
        /*check for supported types*/
        if (regex_compiled && regexec(&supported, name, 0, NULL, 0)) {
            D(g_message("%s not supported",name);)
            continue;
        }
        key = g_strdup(name);
        if (strchr(key,'.')) *strrchr(key,'.')=0;

        /* put in bighash */
        icon_info_p = g_hash_table_lookup(theme_info_p->bighash,key);
        if (!icon_info_p){
	        icon_info_p = (icon_info_t *)malloc(sizeof(icon_info_t));
        } else {
	        icon_info_p->next = (icon_info_t *)malloc(sizeof(icon_info_t));
	        icon_info_p = icon_info_p->next;
        }
        icon_info_p->next=NULL;
        icon_info_p->size=size;
        icon_info_p->type=g_strdup(type);
	icon_info_p->path=g_build_filename(directory,name,NULL);
        if (!g_hash_table_lookup(theme_info_p->bighash,key)){
            D(printf("DBG: putting into bighash: %s\n",key);)
            g_hash_table_insert(theme_info_p->bighash, key, (gpointer) icon_info_p);		
        } else {
            /* key is already in the bighash */
            g_free(key);
            continue;
        }

	    /* put in smallhash */
        {
            GList *list;
            smallhash_t *smallhash_p; 
            for (list=theme_info_p->smallhashes; list; list=list->next){
        	smallhash_p=(smallhash_t *) list->data; 
	    	if (strcmp(smallhash_p->context,context)==0) break;
	    }
	    if (!list){ /* find small hash associated to context */
		smallhash_p = (smallhash_t *)malloc(sizeof(smallhash_t));
		smallhash_p->context = g_strdup(context);
		smallhash_p->hash = g_hash_table_new(g_str_hash, g_str_equal);
		theme_info_p->smallhashes = g_list_append(theme_info_p->smallhashes,smallhash_p);
	    }
	    /* put entry key into smallhash too */
	    if (!g_hash_table_lookup(smallhash_p->hash,key)){
		g_hash_table_insert(smallhash_p->hash, key, (gpointer) icon_info_p);		
	    }
	}
    } /* end while read dir */
    g_dir_close(dir);
    return TRUE;
}

static
gboolean 
add_fallback (gchar *base_dir, const gchar *fallback){
    gchar **p,*directories[]={"48x48/stock/generic",NULL};
    theme_info_t theme_info, *theme_info_p=NULL;
    GList *list;

    if (!g_file_test(base_dir,G_FILE_TEST_IS_DIR)) return FALSE; 
    {
	gchar *g=g_build_filename(base_dir,fallback,NULL);
	if (!g_file_test(g,G_FILE_TEST_IS_DIR)){
	   g_free(g);
	   return FALSE; 
	}
	g_free(g);
    }
    
    theme_info.id=(gchar *)fallback;
    list=g_list_find_custom (theme_list,&theme_info, compare_theme_info);
	
    if (!list){
	list = theme_list= add_theme_to_list(theme_list,fallback);
    }
    if (list) theme_info_p = (theme_info_t *)list->data;
    
    if (theme_info_p) for (p=directories; *p; p++){
	gchar *directory;
	directory=g_build_filename(base_dir,fallback,*p,NULL);
	read_icon_directory(directory,theme_info_p,48,"Threshold","Misc");
	g_free(directory);
    } /* foreach directory */
    return TRUE;
}


static
gboolean 
add_theme_directories (gchar *base_dir, gchar *theme){
    gchar *index=theme_index(base_dir, theme);
    gchar **p,**directories;
    theme_info_t theme_info, *theme_info_p=NULL;
    GList *list;


    XfceRc *themerc;
    if (!index) return FALSE;
	
    if ((directories=get_rc_info(index, "Directories"))==NULL){
	g_free(index);
	return FALSE;
    }
    D(printf("DBG adding directories for %s/%s\n",base_dir,theme);)

    theme_info.id=theme;
    list=g_list_find_custom (theme_list,&theme_info, compare_theme_info);
	
    if (!list){
	g_free(index);
	return FALSE;
    }
    theme_info_p = (theme_info_t *)list->data;

    themerc = xfce_rc_simple_open(index, TRUE);
    if(!themerc) g_assert_not_reached();
    for (p=directories; *p; p++){
	int size;
	const gchar *context, *type;
	gchar *directory;
	
	if(!xfce_rc_has_group(themerc, *p)) continue;
	xfce_rc_set_group(themerc, *p);
	
	/* All Rodent and gnome directories are of type scalable...?*/
	type = xfce_rc_read_entry(themerc, "Type", "Threshold");
	context = xfce_rc_read_entry(themerc, "Context", "Misc");
	size = atoi(xfce_rc_read_entry(themerc, "Size", "0"));

	directory=g_build_filename(base_dir,theme,*p,NULL);
	D(printf("DBG: reading %s...(%s)\n",directory,((directory)?"yes":"no"));)
	read_icon_directory(directory,theme_info_p,size,type,context);
	g_free(directory);
    } /* foreach directory */
    xfce_rc_close(themerc);    
    g_free(index);
    return TRUE;
}


static
void
add_theme_inherits(gchar *base_dir, gchar *theme){
    gchar *index=theme_index(base_dir, theme);
    theme_info_t theme_info;
    if (index) {
	gchar **inherits=NULL;
    
	/* get the inherit themes and add 'em in 
	 * we do not follow inherits from inherits to 
	 * avoid loop conditions from buggy icon.theme files.*/
	if ((inherits = get_rc_info(index,"Inherits"))!= NULL){
	    gchar **p;
	    for (p=inherits; *p; p++) {
		D(printf("DBG: inherits=%s\n",*p);)
		theme_info.id = *p;
		if (!g_list_find_custom (theme_list, &theme_info, compare_theme_info)){
		    theme_list = add_theme_name(theme_list, base_dir, *p);
		    add_theme_directories(base_dir, *p);
		}
		D(else printf("DBG: inherits already listed=%s\n",*p);)
	    }
	    g_strfreev(inherits);
	}	
	g_free(index);
    }
    /* hicolor is fallback for all themes, like it or not... */
    theme_info.id = "hicolor";
    if (!g_list_find_custom (theme_list, &theme_info, compare_theme_info)){
	theme_list = add_theme_name(theme_list, base_dir, theme_info.id);
	add_theme_directories(base_dir, theme_info.id);
    }
}

static
gboolean 
add_theme(GList *list,gchar *theme){
    GList *tmp;
    
    for (tmp=list; tmp; tmp=tmp->next){
	gchar *base_dir=(gchar *)tmp->data;
	theme_list = add_theme_name(theme_list, base_dir, theme);
	add_theme_directories(base_dir, theme);
    }
    
    /* inherit themes should go after all instances of requested theme,
     * that's why two loops...*/
    for (tmp=list; tmp; tmp=tmp->next){
	gchar *base_dir=(gchar *)tmp->data;
	add_theme_inherits(base_dir, theme);	
	add_fallback(base_dir,FALLBACK);
    }
    
    
    return TRUE;
}
    
static
icon_info_t *
find_bighash(const gchar *key){
    icon_info_t *icon_info_p=NULL;
    GList *tmp;
    /* first try theme, then inherits */
    for (tmp=theme_list; tmp; tmp=tmp->next){
	theme_info_t *theme_info_p=(theme_info_t *)tmp->data;
	DD(printf ("DBG:theme=%s\n",theme_info_p->id);)
	icon_info_p = g_hash_table_lookup(theme_info_p->bighash,key);
	if (icon_info_p) break;
    }
    return icon_info_p;
}

static
icon_info_t *
find_smallhash(const gchar *context,const gchar *key){
    icon_info_t *icon_info_p=NULL;
    GList *tmp,*l;
    /* first try theme, then inherits */
    for (tmp=theme_list; tmp; tmp=tmp->next){
	theme_info_t *theme_info_p=(theme_info_t *)tmp->data;
	for (l=theme_info_p->smallhashes; l; l = l->next) {
	    smallhash_t *smallhash_p=(smallhash_t *)l->data;
	    if (strcmp(context,smallhash_p->context)==0){
		icon_info_p = g_hash_table_lookup(smallhash_p->hash,key);
		break;
	    }
	}
	DD(printf ("DBG:theme=%s\n",theme_info_p->id);)
	if (icon_info_p) break;
    }
    return icon_info_p;
}

static
const gchar *
find_icon_path_priv(const gchar *key,int size,const gchar *context){
    icon_info_t *icon_info_p=NULL;
    const gchar *best_match=NULL;
    int best_norm=999999;
    if (!theme_list){
	g_message("Cannot find icon for %s (theme is not open)");
	return NULL;
    }
    if (!key) return NULL;
    if (context) icon_info_p = find_smallhash(context,key);
    if (!icon_info_p) icon_info_p=find_bighash(key);
    if (!icon_info_p) {
	D(g_message("Cannot find icon for %s",key);)
	return NULL;
    }
    /* now select the best match... */
    for (;icon_info_p; icon_info_p = icon_info_p->next){
	int pseudosize;
	int norm;
	/* break on first exact size match, whatever the mimetype,
	 * (svg, png, bmp, xpm, etc) */
	if (size==icon_info_p->size) return (const gchar *)icon_info_p->path;
#if 0
	/* disregard non-scalables (from index.theme definitions)*/
	/* this is currently disabled since some index.theme files
	 * (like gnome) may not use the Threshold type for pngs...  */
	if (strcmp("Scalable",icon_info_p->type)!=0) pseudosize=size;
	  else pseudosize=icon_info_p->size; 
#else
	/* prefer svg icons before any other type (this replaces the
	 * disabled code above)*/
	if (g_str_has_suffix(icon_info_p->path,".svg")) pseudosize=size;
	else pseudosize=icon_info_p->size;
#endif
	/* get closest match */
	norm = abs(size - pseudosize);
	if (!best_match || norm < best_norm) {
	    best_match = icon_info_p->path;
	    best_norm = norm;
	}
    }
    if (!best_match) {
	g_warning("no icon match for %s",key);
    }
    return best_match;
}



/* exported symbols......... */

G_MODULE_EXPORT
int 
open_theme(gchar *theme){
    gboolean result=FALSE;
    static gchar *last_theme=NULL;

    if (!theme) {
#if GTK_CHECK_VERSION (2,4,0)
	g_object_get (gtk_settings_get_default(), "gtk-icon-theme-name", &theme, NULL);
#else /* gtk < 2.4 */
	{
	    GtkSettings *gsettings = gtk_settings_get_default ();
	    g_object_get (G_OBJECT (gsettings), "gtk-icon-theme-name", &theme, NULL);
	}
#endif
    }

    if (!theme) {
        g_warning("no icon theme defined");
        return 0;
    }
    
    if (last_theme && strcmp(last_theme,theme)==0) return -1;
    g_free(last_theme);
    last_theme=g_strdup(theme);
    theme_list = free_theme_list (theme_list);
    base_dirs = get_base_dirs(base_dirs);

    
    
    /* this add inherits too */
    if ((result=add_theme(base_dirs,theme)) == FALSE) {
	return 0; 
    }
#ifdef DEBUG
    showme_themeinfo(theme_list,"theme_list");
#endif

    return 1;
}

G_MODULE_EXPORT
void
close_theme(void) {
    /* free up space taken by lists and hashtables */
    theme_list = free_theme_list (theme_list);
    base_dirs = free_string_list (base_dirs);    
}

G_MODULE_EXPORT
const gchar *
find_icon_path(const gchar *key,int size,const gchar *context){
    gchar *short_key=NULL;
    if (strchr(key,'.')) {
	const gchar *path;
	short_key = g_strdup(key);
	*strrchr(short_key,'.')=0;
	path=find_icon_path_priv(short_key, size, context);
	g_free(short_key);
	return path;
    } else return find_icon_path_priv(key, size, context);
}

#if 0

/* for testing... */
int main(int argc, char ** argv){
    if (argc < 2) {
	g_error("insufficient arguments");
    }
    if (!open_theme(argv[1])){
	g_error("cannot open theme");
    }
    printf("Load ok\n"); 
    while (1) {
	icon_info_t *icon_info_p;
	gchar buf[256];
	printf("key:"); fflush(NULL);
	scanf("%s",buf);
	if (strchr(buf,'\n')) *strchr(buf,'\n')=0;
	icon_info_p=find_smallhash("MimeTypes",buf);
	if (!icon_info_p) icon_info_p=find_bighash(buf);
	if (icon_info_p) {
	    icon_info_t *tmp;
	    for (tmp=icon_info_p; tmp; tmp=tmp->next){
		printf("size=%d,path=%s\n",tmp->size,tmp->path);
	    }
	} else printf("not found\n");
	printf("Best match for size 36 is = %s\n",find_icon_path(buf,36,NULL));
	printf("Best match for size 48 is = %s\n",find_icon_path(buf,48,NULL));
	printf("Best match for size 58 is = %s\n",find_icon_path(buf,58,NULL));
    }
}

#endif
