/*
 * Copyright (c) 2004 Jean-Franois Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <gtk/gtkversion.h>
#if GTK_CHECK_VERSION(2, 4, 0)
#undef GTK_DISABLE_DEPRECATED
#endif

/* */
#include <libxfce4mcs/mcs-manager.h>
#include <libxfcegui4/libxfcegui4.h>
#include <xfce-mcs-manager/manager-plugin.h>

#include <printing-system.h>

/* */
#define RCDIR           "mcs_settings"
#define OLD_RCDIR       "settings"
#define CHANNEL         "xfprint"
#define RCFILE          "xfprint.xml"

#define SOEXT 		("." G_MODULE_SUFFIX)
#define SOEXT_LEN 	(strlen (SOEXT))
#define strequal(s1,s2) !strcmp (s1, s2)

#define BORDER          6

enum {
  COLUMN_NAME,
  COLUMN_PATH,
  COLUMN_CLASS,
  N_COLUMNS
};
/* setting */
static gchar *printing_system = NULL;

/* */
static gboolean is_running = FALSE;

typedef struct _Itf Itf;
struct _Itf {
  McsPlugin *plugin;

  GtkWidget *dialog;
  GtkWidget *treeview;
  GtkWidget *description;
  GtkWidget *version;
  GtkWidget *author;
  GtkWidget *homepage;
};

/* static prototypes */
static void run_dialog (McsPlugin *);
static gboolean save_settings (McsPlugin *);

/* */
McsPluginInitResult mcs_plugin_init (McsPlugin *plugin)
{
  McsSetting *setting;
  gchar *file, *path;
  
  xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

  /* read settings channel from file */
  path = g_build_filename ("xfce4", RCDIR, RCFILE, NULL);
  file = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, path);
  g_free (path);

  if (!file)
    file = xfce_get_userfile (OLD_RCDIR, RCFILE, NULL);

  if (g_file_test (file, G_FILE_TEST_EXISTS)) {
    mcs_manager_add_channel_from_file (plugin->manager, CHANNEL, file);
  } else {
    mcs_manager_add_channel (plugin->manager, CHANNEL);
  }
  
  g_free (file);
  
  if ((setting = mcs_manager_setting_lookup (plugin->manager,
					     "XfPrint/system",
					     CHANNEL)) != NULL) {
    printing_system = g_strdup (setting->data.v_string);
  } else {
    printing_system = g_build_filename (PLUGIN_DIR, "file_plugin.so", NULL);
    mcs_manager_set_string (plugin->manager, "XfPrint/system",
			    CHANNEL, printing_system);
  }

  plugin->plugin_name = g_strdup ("printing system");
  plugin->caption = g_strdup (_("Printing system"));
  plugin->run_dialog = run_dialog;
  plugin->icon = xfce_themed_icon_load ("xfprint", 48);
  mcs_manager_notify (plugin->manager, CHANNEL);
  
  save_settings (plugin);
  
  return (MCS_PLUGIN_INIT_OK);
}


static gboolean save_settings (McsPlugin *plugin)
{
  gboolean result;
  gchar *file, *path;

  path = g_build_filename ("xfce4", RCDIR, RCFILE, NULL);
  file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, path, TRUE);
  
  result = mcs_manager_save_channel_to_file (plugin->manager, CHANNEL,
					     file);
  g_free(path);
  g_free(file);

  return(result);
}

/* Callbacks */
static gboolean psinfos_free_foreach_func (GtkTreeModel *model,
					   GtkTreePath *path,
					   GtkTreeIter *iter,
					   gpointer data)
{
  PrintingSystemInfo *psinfo;

  gtk_tree_model_get (model, iter, COLUMN_CLASS, &psinfo, -1);
  printing_system_info_free (psinfo);

  return FALSE;
}

static void cb_dialog_response(GtkWidget * dialog, gint response_id, gpointer data)
{
  Itf *itf;

  itf = (Itf *) data;

  if(response_id == GTK_RESPONSE_CLOSE){
    GtkTreeModel *model;

    is_running = FALSE;

    /* free psclasses */
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (itf->treeview));
    gtk_tree_model_foreach (model, psinfos_free_foreach_func,
			    itf);

    gtk_widget_destroy(dialog);
  }
}

static void cb_selection_changed (GtkTreeSelection *selection, gpointer data)
{
  Itf *itf;
  GtkTreeModel *model;
  GtkTreeIter   iter;

  itf = (Itf *) data;

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    PrintingSystemInfo *psinfo;
    gchar *path;

    gtk_tree_model_get (model, &iter, COLUMN_CLASS, &psinfo,
			COLUMN_PATH, &path, -1);

    gtk_label_set_text (GTK_LABEL (itf->description), psinfo->description);
    gtk_label_set_text (GTK_LABEL (itf->version), psinfo->version);
    gtk_label_set_text (GTK_LABEL (itf->author), psinfo->author);
    gtk_label_set_text (GTK_LABEL (itf->homepage), psinfo->homepage);

    if (strcmp (psinfo->description, _("None")) == 0)
      gtk_widget_set_sensitive (itf->description, FALSE);
    else
      gtk_widget_set_sensitive (itf->description, TRUE);
    if (strcmp (psinfo->version, _("None")) == 0)
      gtk_widget_set_sensitive (itf->version, FALSE);
    else
      gtk_widget_set_sensitive (itf->version, TRUE);
    if (strcmp (psinfo->author, _("None")) == 0)
      gtk_widget_set_sensitive (itf->author, FALSE);
    else
      gtk_widget_set_sensitive (itf->author, TRUE);
    if (strcmp (psinfo->homepage, _("None")) == 0)
      gtk_widget_set_sensitive (itf->homepage, FALSE);
    else
      gtk_widget_set_sensitive (itf->homepage, TRUE);

    g_free (printing_system);
    printing_system = g_strdup (path);

    mcs_manager_set_string (itf->plugin->manager, "XfPrint/system",
			    CHANNEL, printing_system);
    mcs_manager_notify (itf->plugin->manager, CHANNEL);
    save_settings (itf->plugin);

    g_free (path);
  }
}

/* */
static Itf *create_xfprint_dialog (McsPlugin * plugin)
{
  Itf *dialog;
  GtkWidget *header;
  GtkWidget *hbox;
  GtkWidget *swin;
  GtkListStore *store;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;

  dialog = g_new (Itf, 1);

  dialog->plugin = plugin;

  dialog->dialog = gtk_dialog_new ();
  gtk_window_set_icon (GTK_WINDOW (dialog->dialog), plugin->icon);
  gtk_window_set_title (GTK_WINDOW (dialog->dialog), _("Printing system"));
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog->dialog), FALSE);

  header = xfce_create_header (plugin->icon, _("Printing System Selection"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox),
		      header, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, BORDER);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox), hbox,
                      TRUE, TRUE, BORDER);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER - 1);

  /* add printing systems list */
  swin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin), 
                                  GTK_POLICY_NEVER,
                                  GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (swin),
                                       GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (hbox), swin, FALSE, FALSE, 0);

  store = gtk_list_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING, 
			      G_TYPE_POINTER);
  dialog->treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (dialog->treeview), FALSE);
  gtk_container_add (GTK_CONTAINER (swin), dialog->treeview);
  g_object_unref (G_OBJECT (store));

  column = gtk_tree_view_column_new ();
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, FALSE);
  gtk_tree_view_column_set_attributes (column, renderer,
                                       "text", COLUMN_NAME,
                                       NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (dialog->treeview), column);

  /* add information frame */
  frame = gtk_frame_new (_("Information"));
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);
  gtk_container_set_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_widget_show (table);

  label = gtk_label_new (_("<b>Description:</b>"));
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_table_attach (GTK_TABLE (table), label,
                    0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  dialog->description = gtk_label_new ("");
  gtk_label_set_line_wrap (GTK_LABEL (dialog->description), TRUE);
  gtk_label_set_selectable (GTK_LABEL (dialog->description), TRUE);
  gtk_misc_set_alignment (GTK_MISC (dialog->description), 0, 0);
  gtk_table_attach (GTK_TABLE (table), dialog->description,
                    1, 2, 0, 1, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new (_("<b>Version:</b>"));
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_table_attach (GTK_TABLE (table), label,
                    0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);

  dialog->version = gtk_label_new ("");
  gtk_label_set_selectable (GTK_LABEL (dialog->version), TRUE);
  gtk_misc_set_alignment (GTK_MISC (dialog->version), 0, 0);
  gtk_table_attach (GTK_TABLE (table), dialog->version,
                    1, 2, 1, 2, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new (_("<b>Author:</b>"));
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_table_attach (GTK_TABLE (table), label,
                    0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 0);

  dialog->author = gtk_label_new ("");
  gtk_label_set_selectable (GTK_LABEL (dialog->author), TRUE);
  gtk_misc_set_alignment (GTK_MISC (dialog->author), 0, 0);
  gtk_table_attach (GTK_TABLE (table), dialog->author,
                    1, 2, 2, 3, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new (_("<b>Homepage:</b>"));
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_table_attach (GTK_TABLE (table), label,
                    0, 1, 3, 4, GTK_FILL, GTK_FILL, 0, 0);

  dialog->homepage = gtk_label_new ("");
  gtk_label_set_selectable (GTK_LABEL (dialog->homepage), TRUE);
  gtk_misc_set_alignment (GTK_MISC (dialog->homepage), 0, 0);
  gtk_table_attach (GTK_TABLE (table), dialog->homepage,
                    1, 2, 3, 4, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  /* add close button */
  button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog->dialog), button,
				GTK_RESPONSE_CLOSE);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);

  return dialog;
}

/* */
static void setup_dialog(Itf * itf)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GDir *gdir;
  const char *file;

  GtkTreeSelection *selection;

  /* Connect callbacks */
  g_signal_connect (G_OBJECT (itf->dialog), "response",
		    G_CALLBACK (cb_dialog_response), itf);
  g_signal_connect (G_OBJECT (itf->dialog), "delete-event",
                    G_CALLBACK (cb_dialog_response), itf);

  /* handle selection */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (itf->treeview));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (cb_selection_changed), itf);

  /* Initialize printing system list */
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (itf->treeview));

  gdir = g_dir_open (PLUGIN_DIR, 0, NULL);
  if (gdir) {
    while ((file = g_dir_read_name (gdir))) {
      const char *s = file;
    
      s += strlen (file) - SOEXT_LEN;
      
      if (strequal (s, SOEXT)) {
	char *path;
	GModule *gm;
	gpointer symbol;
	PrintingSystemInfo *psinfo;

	path = g_build_filename (PLUGIN_DIR, file, NULL);

	gm = g_module_open (path, 0);

	if(gm && g_module_symbol (gm, "printing_system_info_init_impl", &symbol)){
	  void (*init) (PrintingSystemInfo * psinfo);

	  psinfo = g_new0 (PrintingSystemInfo, 1);

	  init = symbol;
	  init (psinfo);

	  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
	  gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			      COLUMN_NAME, psinfo->name,
			      COLUMN_PATH, path,
			      COLUMN_CLASS, psinfo, -1);
	  if (strcmp (path, printing_system) == 0){
	    GtkTreePath *treepath;

	    treepath = gtk_tree_model_get_path (GTK_TREE_MODEL (model), &iter);
	    gtk_tree_view_set_cursor (GTK_TREE_VIEW (itf->treeview), treepath, NULL, FALSE);
	    gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (itf->treeview), treepath, NULL,
                                TRUE, 0.5, 0.0);
	    gtk_tree_path_free (treepath);
	  }

	  g_module_close (gm);
	}else if(gm){
	  g_warning ("No 'printing_system_class_init_impl' function found in %s", path);
	  g_module_close (gm);
	}

	g_free (path);
      }
    }

  }
  g_dir_close (gdir);

  /* Show dialog */
  gtk_window_set_position (GTK_WINDOW (itf->dialog),
			   GTK_WIN_POS_CENTER);
  gtk_widget_show_all (itf->dialog);
}

/* */
static void run_dialog (McsPlugin *plugin)
{
  Itf *dialog;

  if(!is_running){
    is_running = TRUE;
    
    xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

    dialog = create_xfprint_dialog (plugin);
    setup_dialog (dialog);
  }
}

/* */
MCS_PLUGIN_CHECK_INIT

