/*
 *
 * Edscott Wilson Garcia 2001-2004 for xfce project.
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>

#include <errno.h>
#include <limits.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <glib.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <gmodule.h>
#include <libxfce4util/libxfce4util.h>

/* include non installed headers */
#include <xfce4-modules/headers/constants.h>
#include <xfce4-modules/headers/mime.h>

/*#define DEBUG
#define D(x) x*/
#ifndef HAVE_LIBXML
#warning "mime_icons module will be disabled without libxml!"
G_MODULE_EXPORT
const gchar *g_module_check_init(GModule *module){
    g_error("Cannot work with mime-icons module: libxfcegui4 constructed without libxml.");
    return NULL;
}
#else

typedef struct magic_t {
	char *mimetype;
	char *type;
	char *value;
	char *mask;
	int offset;
	int last_offset;
	int priority;
}magic_t;

static int magic_max = 16; /* this would be host32, to be superceded by string lengths */

G_MODULE_EXPORT
void mime_add(const gchar *sfx, const gchar *command);
G_MODULE_EXPORT
gboolean is_valid_command(const char *cmd_fmt);

G_MODULE_EXPORT
const gchar *mime_typeinfo(const gchar *type);
G_MODULE_EXPORT
const gchar *mime_key_type (const gchar *file);
G_MODULE_EXPORT
const gchar *mime_command(const gchar *file);
G_MODULE_EXPORT
const gchar **mime_apps(const gchar *file);
G_MODULE_EXPORT
const gchar *mk_command_line(const gchar *command_fmt,const gchar *path,gboolean interm,gboolean hold);
G_MODULE_EXPORT
const gchar *mime_get_type(const gchar *file, gboolean try_magic);

static GHashTable *application_hash = NULL;
static GList *magic_list = NULL;
static xfmime_functions *xfmime_fun=NULL;

static  mime_t *
locate_mime_t(const gchar *file);

G_MODULE_EXPORT
const gchar *g_module_check_init(GModule *module){
    xfmime_fun = g_new0 (xfmime_functions,1);
    if (!xfmime_fun) return ("unable to create function structure");
    
    xfmime_fun->mime_command = mime_command;
    xfmime_fun->mime_typeinfo = mime_typeinfo;
    xfmime_fun->mime_key_type = mime_key_type;
    xfmime_fun->mime_get_type = mime_get_type;
    xfmime_fun->mime_apps = mime_apps;
    xfmime_fun->mime_add = mime_add;
    xfmime_fun->mk_command_line = mk_command_line;
    xfmime_fun->is_valid_command = is_valid_command;
    return NULL;
}



static void clear_application(gpointer key, gpointer value, gpointer user_data)
{
    int i;
    mime_t *mime;
    mime = (mime_t *)value;
    if (mime->apps) for (i=0;mime->apps[i];i++) g_free(mime->apps[i]);
    g_free(mime->apps);
    g_free(mime->key);
    g_free(mime->mimetype);
    g_free(mime); mime=NULL;
}

static void destroy_application_hash(GHashTable *hash){
    if (!hash) return;
    g_hash_table_foreach(hash, clear_application, NULL);
    g_hash_table_destroy(hash);
}

static void mime_build_magic(void){
 xmlNodePtr node;
 xmlNodePtr subnode,ssnode;
 xmlDocPtr doc;
 gchar *mimefile=NULL;
    
#ifdef DEBUG
            printf("DBG: executing mime_build_magic...\n");
#endif
	 
   	mimefile=g_build_filename(MAGIC_MIME_FILE,NULL);
    
#ifdef DEBUG
            printf("DBG: mimefile=%s\n",mimefile);
#endif
    if (access(mimefile,R_OK)!=0){
	g_free(mimefile); mimefile=NULL;
#ifdef DEBUG
            printf("DBG: access(mimefile,R_OK)!=0\n");
#endif
	return;
    }
    xmlKeepBlanksDefault(0);

    if((doc = xmlParseFile(mimefile)) == NULL)
    {
#ifdef DEBUG
      	printf("xfce4-modules: invalid xml file: %s\n",mimefile);
#endif
	g_free(mimefile); mimefile=NULL;
	return;
    }
	
    node = xmlDocGetRootElement(doc);
    if(!xmlStrEqual(node->name, (const xmlChar *)"mime-info"))
    {
#ifdef DEBUG
      	printf("xfce4-modules: invalid xml file %s\n",mimefile);
#endif
	g_free(mimefile); mimefile=NULL;
        xmlFreeDoc(doc);
	return;
    }
    
    /* Now parse the xml tree */
#ifdef DEBUG
            printf("DBG: parsing %s\n",mimefile);
#endif
    for(node = node->children; node; node = node->next)
    {
	if(xmlStrEqual(node->name, (const xmlChar *)"mime-type"))
	{
	    magic_t *magic;
	    xmlChar *mimetype;
	    mimetype = (char *)xmlGetProp(node, (const xmlChar *)"type");
	    if (!mimetype) continue;
	    for(subnode = node->children; subnode; subnode = subnode->next){
		if(xmlStrEqual(subnode->name, (const xmlChar *)"magic")){
 		   xmlChar *priority = xmlGetProp(subnode, (const xmlChar *)"priority");
		   int pri=(priority)?atoi(priority):0;
		   if (priority) g_free(priority);
	           for(ssnode = subnode->children; ssnode; ssnode = ssnode->next){
		      if(xmlStrEqual(ssnode->name, (const xmlChar *)"match")){
 		        xmlChar *type,*offset,*value,*mask;
			
	   		magic = (magic_t *)malloc(sizeof(magic_t));
	    		magic->mimetype = g_strdup(mimetype);
			
	    		magic->priority = pri;
	    		magic->value = NULL;
	    		magic->type = NULL;
	    		magic->mask = NULL;
	    		magic->offset = 0;

	                type   = xmlGetProp(ssnode, (const xmlChar *)"type");
	                value  = xmlGetProp(ssnode, (const xmlChar *)"value");
	                offset = xmlGetProp(ssnode, (const xmlChar *)"offset");
	                mask   = xmlGetProp(ssnode, (const xmlChar *)"mask");
		        if (type) {
				magic->type = g_strdup(type);
			       g_free(type);	
			} else magic->priority=0;
		        if (value){
			       if (strlen(value) > magic_max) magic_max = strlen(value);
			       magic->value = g_strdup(value);
			       g_free(value);
			} else magic->priority=0;
			if (offset){
				if (strchr(offset,':')) {
					char *c = strtok(offset,":");
					magic->offset = atoi(c);
					c = strtok(NULL,":");
					magic->last_offset = atoi(c);
				} else {
					magic->last_offset = magic->offset = atoi(offset); 
				}
				g_free(offset);
			} else magic->priority=0;
			if (mask){
				magic->mask = g_strdup(mask);
				g_free(mask);
			}
	    		magic_list = g_list_append(magic_list,(gpointer)magic);
#ifdef DEBUG
			if (magic->priority != pri) printf(" Priorities do not match!!!\n");
			printf("inserting: %s=%d {type=%s value=%s offset=%d mask=%s}\n",
				magic->mimetype,magic->priority, 
				magic->type,magic->value,magic->offset,
				magic->mask?magic->mask:"");
#endif
	
		      } /* match */
		   }/* ssnode */
		}/* magic */
	    }/* subnode */			
	    g_free(mimetype);
 	}/* mime-type */
    } /* mime-info */
    xmlFreeDoc(doc);
    g_free(mimefile);
}

static void mime_build_list(void){
 xmlChar *value;
 xmlNodePtr node;
 xmlNodePtr subnode;
 xmlDocPtr doc;
 gchar *mimefile=NULL;
 int i;
    
#ifdef DEBUG
            printf("DBG: executing mime_build_list...\n");
#endif
 if (application_hash){
    destroy_application_hash(application_hash);
    application_hash=NULL;
 }

 application_hash = g_hash_table_new(g_str_hash, g_str_equal);

 for (i=0;i<2;i++) {
	 
    if (i) {
   	mimefile=g_build_filename(APPLICATION_MIME_FILE,NULL);
    } else {
	gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	if (!g_get_home_dir()) continue;
       	mimefile=g_build_filename(xdg_dir,LOCAL_APPLICATION_MIME_FILE,NULL);
	g_free(xdg_dir);
    }
#ifdef DEBUG
            printf("DBG: mimefile=%s\n",mimefile);
#endif
    if (access(mimefile,R_OK)!=0){
	g_free(mimefile); mimefile=NULL;
#ifdef DEBUG
            printf("DBG: access(mimefile,R_OK)!=0\n");
#endif
	continue;
    }
    xmlKeepBlanksDefault(0);

    if((doc = xmlParseFile(mimefile)) == NULL)
    {
	gchar *g = g_strconcat(mimefile,".bak");
#ifdef DEBUG
      	printf("xfce4-modules: invalid xml file %s.bak\n",mimefile);
#endif
	rename(mimefile,g);
	g_free(g);
	g_free(mimefile); mimefile=NULL;
	continue;
    }
	
    node = xmlDocGetRootElement(doc);
    if(!xmlStrEqual(node->name, (const xmlChar *)"mime-info"))
    {
	gchar *g = g_strconcat(mimefile,".bak");
#ifdef DEBUG
      	printf("xfce4-modules: invalid xml file %s.bak\n",mimefile);
#endif
	rename(mimefile,g);
	g_free(g);
	g_free(mimefile); mimefile=NULL;
        xmlFreeDoc(doc);
	continue;
    }
    /* Now parse the xml tree */
#ifdef DEBUG
            printf("DBG: parsing %s\n",mimefile);
#endif
    for(node = node->children; node; node = node->next)
    {
	if(xmlStrEqual(node->name, (const xmlChar *)"mime-key"))
	{
	    mime_t *mime;
	    char *okey;
	    gchar *key;
	    okey = (char *)xmlGetProp(node, (const xmlChar *)"key");
	    if (!okey) continue;
	    key=g_utf8_strdown (okey,-1);
	    g_free(okey);
	    okey=NULL;
    	    mime = g_hash_table_lookup(application_hash, key);

	    if (!mime) {
		mime=(mime_t *)malloc(sizeof(mime_t));
	    	if (!mime) g_assert_not_reached();
	    	mime->apps = (char **)malloc(1*sizeof(char *));
	    	*(mime->apps)=NULL;
		mime->key = key;
	    } else {
		g_free(key);
		key=NULL;
		/*override current type*/
		g_free(mime->mimetype);
	    }
	    mime->mimetype = (char *)xmlGetProp(node, (const xmlChar *)"type");
	    	    
	    for(subnode = node->children; subnode; subnode = subnode->next){
		if(xmlStrEqual(subnode->name, (const xmlChar *)"application")){
	           value = xmlGetProp(subnode, (const xmlChar *)"command");
		   if (value){
		     int i,j;
		     char **tmp=mime->apps;
		     for (i=0;mime->apps[i];i++){
			     if (strcmp(value,mime->apps[i])==0){
				     i=-1;
				     break;
			     }
		     }
		     if (i >= 0){
		       mime->apps = (char **)malloc((i+2)*sizeof(char *));
		       for (j=0;j<i;j++){
			     mime->apps[j] = tmp[j];
		       }
		       *(mime->apps+i)  = (char *)value;
		       *(mime->apps+i+1) = NULL;
		       g_free(tmp);
#ifdef DEBUG
	               printf("DBG:adding : %s for %s\n",*(mime->apps+i),mime->key);
#endif 		
		     }
		   }
#ifdef DEBUG
		   else
	             printf("DBG:Null command for %s\n",mime->mimetype);		    
#endif 		   
		}
	    }
	    g_hash_table_replace (application_hash,(gpointer)mime->key,(gpointer)mime);
	}
    }
    xmlFreeDoc(doc); 
    g_free(mimefile);
 }	
}


static const gchar *mimeable_file(const gchar *file){
	struct stat st;
	if (stat(file,&st)<0) {
		/*printf("stat(file,&st)<0!\n");*/
		return NULL;
	}
#ifdef S_IFWHT
	if (st.st_mode == S_IFWHT){
		/*printf("S_IFWHT file!\n");*/
	      	return NULL;
	}
#endif
	if (S_ISSOCK(st.st_mode) ) return "inode/socket";
	if (S_ISBLK(st.st_mode) ) return "inode/blockdevice";
	if (S_ISCHR(st.st_mode) ) return "inode/chardevice";
	if (S_ISFIFO(st.st_mode) ) return "inode/fifo";
	if (S_ISLNK(st.st_mode)) return "inode/symlink";/* won't happen */
	if (S_ISDIR(st.st_mode)) return "inode/directory";
	return "";
}


G_MODULE_EXPORT
const char *mime_key_type (const gchar *file){
	mime_t *mime;
	const gchar *m=mimeable_file(file);
	if (!m || strlen(m)) return NULL;
  	if (!application_hash) mime_build_list();
	mime= locate_mime_t(file);
	if (!mime || !mime->mimetype) return NULL;
	return mime->mimetype;
}


static void mime_write(char *sfx,mime_t *mime){
    xmlNodePtr node,root;
    xmlNodePtr subnode;
    xmlDocPtr doc;
    char *mimefile=NULL;
    xmlChar *oldcommand = NULL;
    
    if (!g_get_home_dir()) return;
    if (!mime || !mime->apps || !mime->apps[0]) return;
    {
	gchar *xdg_dir=	xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	mimefile=g_build_filename(xdg_dir,LOCAL_APPLICATION_MIME_FILE,NULL);
	g_free(xdg_dir);
    }

    if (access(mimefile,F_OK)==0 && access(mimefile,W_OK)!=0){
	g_warning("cannot write to %s", mimefile);
	g_free(mimefile); mimefile=NULL;
	return;
    }
    
    /* read xml in */
    if (access(mimefile,R_OK)!=0){
       doc = xmlNewDoc("1.0");
       doc->children = xmlNewDocRawNode(doc, NULL, "mime-info", NULL);

       root = (xmlNodePtr) doc->children;
       xmlDocSetRootElement(doc, root);
       node = xmlNewTextChild(root, NULL, "mime-key", NULL);
    } else {
       xmlChar *value;
       if((doc = xmlParseFile(mimefile)) == NULL)
       {
         g_warning("xfce4-modules: invalid xml file %s.",mimefile);
	 g_free(mimefile); mimefile=NULL;
	 return;
       }
       root = xmlDocGetRootElement(doc);
       if(!xmlStrEqual(root->name, (const xmlChar *)"mime-info"))
       {
         g_warning("xfce4-modules: invalid xml file %s.",mimefile);
	 g_free(mimefile); mimefile=NULL;
         xmlFreeDoc(doc);
	 return;
       }
       for (node=root->children; node; node=node->next){
#ifdef DEBUG
	       printf("DBG: %s == extension ?\n",node->name);
#endif
	       if(xmlStrEqual(node->name, (const xmlChar *)"mime-key")){
	          value = xmlGetProp(node, (const xmlChar *)"key");
		  if (value){
#ifdef DEBUG
			  printf("DBG: %s == %s ?\n",value,sfx);
#endif
			  if (xmlStrEqual((const xmlChar *)sfx, (const xmlChar *)value)){
				  g_free(value);
				  break;
			  }
			  g_free(value);
		  }
	       }
       }
       if (!node) {
	       	node = xmlNewTextChild(root, NULL, "mime-key", NULL);
    		xmlSetProp(node,"key", sfx);
    		xmlSetProp(node,"type", mime->mimetype);
       }
    }   

    if (node->children){
       for (subnode=node->children; subnode; subnode=subnode->next){
	   if (xmlStrEqual(subnode->name, (const xmlChar *)"application")){
		gchar *g = g_strescape ((const gchar *)(mime->apps[0]),"\f\n\r");
		oldcommand = xmlGetProp(subnode, (const xmlChar *)"command");
		xmlSetProp(subnode,"command", g);
		g_free(g);
		break;
	   }
       }
    }  
		      
    /* add additional mime register */
    subnode = xmlNewTextChild(node, NULL, "application", NULL);
    {
	gchar *g;
        if (oldcommand) {
		g = g_strescape ((const gchar *)(oldcommand),"\f\n\r");
		g_free(oldcommand);
	}
	else g = g_strescape ((const gchar *)(mime->apps[0]),"\f\n\r");
	xmlSetProp(subnode,"command", g);
	g_free(g);
    }
    
    xmlSaveFormatFile(mimefile, doc, 1);

    xmlFreeDoc(doc);
    g_free(mimefile);
    return;
}

G_MODULE_EXPORT
void mime_add(const gchar *file, const gchar *command)
{
    gchar *sfx;
    mime_t *mime;
    if (!command || !strlen(command)) return;
    if (!application_hash) mime_build_list();

    if (strrchr(file,G_DIR_SEPARATOR)) sfx = strrchr(file,G_DIR_SEPARATOR)+1;
    else sfx=(gchar *)file;
    if (strchr(sfx, '.')) sfx=strchr(sfx, '.')+1;
    /*printf("adding suffix %s\n",sfx);*/
    if (!sfx || !strlen(sfx)) return;
    sfx = g_utf8_strdown (sfx,-1);
    
    mime = g_hash_table_lookup(application_hash, sfx);

    if (!mime) {
	mime=(mime_t *)malloc(sizeof(mime_t));
    	if (!mime) g_assert_not_reached();
    	mime->apps = (char **)malloc(2*sizeof(char *));
    	*(mime->apps)=g_strdup(command);
    	*(mime->apps+1)=NULL;
	mime->key = g_strdup(sfx);
	mime->mimetype = g_strconcat("application/xffm-",sfx,NULL);
    } else {
       int i,j;
       char **tmp=mime->apps;
       for (i=0;mime->apps[i];i++);
       mime->apps = (char **)malloc((i+2)*sizeof(char *));
       *(mime->apps) = g_strdup(command);
       for (j=0; j<i; j++) mime->apps[j+1] = tmp[j];
       *(mime->apps+i+1) = NULL;
       g_free(tmp);
    }
    g_hash_table_replace(application_hash,  (gpointer) (mime->key),(gpointer) mime);
    mime_write(sfx,mime);
    g_free(sfx);
    return;
}

G_MODULE_EXPORT
gboolean is_valid_command(const char *cmd_fmt){
       GError *error = NULL;
       int argc;
       gchar *path;
       gchar **argv;
       if (!cmd_fmt) return FALSE;
       if (!g_shell_parse_argv (cmd_fmt, &argc,&argv,&error)){
	  gchar *msg = g_strcompress (error->message);
	  g_warning("%s: %s",msg,cmd_fmt);
	  g_error_free (error);
	  g_free (msg);
	  return FALSE;
       }
       path=g_find_program_in_path(argv[0]);
       if (!path || access(path,X_OK)!=0){
         g_strfreev (argv);
	 if (!path) errno=ENOENT;
         return FALSE;
       }
       g_strfreev (argv);
       return TRUE;
}

G_MODULE_EXPORT
const gchar *mk_command_line(const gchar *command_fmt,const gchar *path,gboolean interm,gboolean hold){
	static gchar *command_line=NULL;
	gchar  *fmt=NULL,*termcmd=NULL;
	gboolean quoted=FALSE;
	gchar *p;
	
	if (!command_fmt) return NULL;
	if (command_line){
		g_free(command_line); 
		command_line=NULL;
	}
	if (!path) path="";

	if(interm){
	    	gchar *term=NULL;
	    	if(getenv("TERMCMD") && strlen( getenv("TERMCMD"))){
		    term = g_strdup(getenv("TERMCMD"));
	    	} else {
		    term = g_strdup("xterm");
	    	}
		if (!is_valid_command(term)){
			g_warning("%s == NULL",term);
			g_free(term);
			return NULL;
		}
		if (hold && strncmp(term,"xterm",strlen("xterm"))==0) {
			termcmd=g_strconcat(term," -hold -e ",NULL);
		} else {
			termcmd=g_strconcat(term," -e ",NULL);
		}
		g_free(term);
	}
	
	D(printf("DBG: command_fmt=%s\n",command_fmt);)

	/* this is to send path as an argument */
	    
	if (strstr(command_fmt,"\%s")){
		fmt = g_strconcat(((termcmd)?termcmd:""),command_fmt,NULL);
	} else {
		fmt = g_strconcat(((termcmd)?termcmd:""),command_fmt," \%s",NULL);
	}
	D(printf("DBG: fmt=%s\n",fmt);)

        for (p=(gchar *)path;*p;p++){
	   if ((*p>='A' && *p<='Z') || (*p>='a' && *p<='z') || (*p>='0' && *p<='9')) continue;
	    quoted=TRUE;
	    break;
	}
	if (!quoted) command_line = g_strdup_printf(fmt,path);
	else {
		gchar *quoted_path=g_strdup_printf("\"%s\"",path);
		command_line = g_strdup_printf(fmt,quoted_path);
		g_free(quoted_path);
	}
	g_free(fmt);
	g_free(termcmd);
	return command_line;
}

G_MODULE_EXPORT
const gchar *mime_command(const gchar *file){
  mime_t *mime;
  int i;
  static gchar *cmd_fmt=NULL;
  g_free(cmd_fmt); cmd_fmt=NULL;
  if (!application_hash) mime_build_list();
  mime=locate_mime_t (file);
  if (!mime || !mime->apps || !mime->apps[0]) return NULL;
  for (i=0; mime->apps[i]; i++){
    g_free(cmd_fmt);
    cmd_fmt = g_strcompress(mime->apps[i]);
    if (is_valid_command(cmd_fmt)) return cmd_fmt;	  
  }
  g_free(cmd_fmt); cmd_fmt=NULL;
  D(printf("DBG: mime_command(%s)=%s\n",file,((cmd_fmt)?cmd_fmt:"null"));)
  return (const gchar *)cmd_fmt;
}

G_MODULE_EXPORT
const gchar **mime_apps(const gchar *file){
  mime_t *mime;
  if (!application_hash) mime_build_list();
  mime=locate_mime_t (file);
  if (!mime || !mime->apps) return NULL;
  return (const gchar **)mime->apps;
}

G_MODULE_EXPORT
xfmime_functions *module_init(void){
    return xfmime_fun;
}

static void apply_string_mask(unsigned char *s, char *mask){
	/*FIXME(maybe): apply mask, format is hexadecimal base, 
	 * like "0xffffffff000000000000000000000000ff"
	 * */
	return;
}

static gboolean string_compare(unsigned char *value,unsigned char *t,char *mask){
	unsigned char *c;
	int i,j;
	gboolean result=FALSE;
	/* overkill, but safe: */
	c = (unsigned char *) malloc(4*strlen(value)+1);

	for (j=i=0;i<strlen(value);i++,j++){
		if (t[i]<128 && t[i]>32)sprintf(c+j,"%c",*(t+i));
		else {
			sprintf(c+j,"\\%03d",*(t+i));
			j += 3;
		}
	}
	if (j > 4*strlen(value)){
		g_warning("mime.c: at string_compare(), j > 4*strlen(value) (%d > %d)\n Expect a SegV error!",
				j,4*strlen(value));
	}
	c[j]=0;
	
	
	apply_string_mask(c, mask);/* FIXME*/
	/*D(printf("%s == %s\n",value,c);)*/
	if (strcmp(value,c)==0) result=TRUE;
	g_free(c);
	return result;
}

/* big16 or little16 */

#if SIZEOF_SHORT==2
#define int16 short int
#else
#warning "short int > 2! Some mime types might not be magic detected."
#define int16 short int
#endif

#if SIZEOF_SHORT==4
#define int32 short int
#endif

#if SIZEOF_INT==4
#define int32 int
#endif


static gboolean host16_compare(unsigned char *value,unsigned char *t,char *mask){
	int16 i;
	long k = strtol(value,NULL,0);
	unsigned char *c=(unsigned char *)&i;
#ifdef WORDS_BIGENDIAN	
	c[0] = *(t);
	c[1] = *(t+1);

#else
	c[1] = *(t);
	c[0] = *(t+1);
#endif
	if (k == i) return TRUE;
	return FALSE;
}	

static gboolean host32_compare(unsigned char *value,unsigned char *t,char *mask){
	int32 i;
	long k = strtol(value,NULL,0);
	unsigned char *c=(unsigned char *)&i;
#ifdef WORDS_BIGENDIAN	
	c[0] = *(t);
	c[1] = *(t+1);
	c[2] = *(t+2);
	c[3] = *(t+3);
#else
	c[3] = *(t);
	c[2] = *(t+1);
	c[1] = *(t+2);
	c[0] = *(t+3);
#endif
	if (k==i) return TRUE;
	return FALSE;
}

static gboolean int16_compare(unsigned char *value,unsigned char *t,char *mask){
	int16 *i=(int16 *)t;
	long k = strtol(value,NULL,0);
	if (k==*i) return TRUE;
	return FALSE;
}	

static gboolean int32_compare(unsigned char *value,unsigned char *t,char *mask){
	int32 *i=(int32 *)t;
	long k = strtol(value,NULL,0);
	if (k==*i) return TRUE;
	return FALSE;
}

static gboolean byte_compare(unsigned char *value,unsigned char *t,char *mask){
	long k = strtol(value,NULL,0);
	if (k == *t) return TRUE;
	return FALSE;
}	


static const gchar *last_type_determination(const gchar *file){
	/*const gchar *result;*/
	FILE *f;
	unsigned char buffer[256];
	size_t i,b;

	/* certain back up types? */
	if (file[strlen(file)-1] == '~') return "application/x-trash";
	/* xffm backup type? */
	if (strrchr(file,'-')){
	    gchar *p=strrchr(file,'-')+1;
	    gboolean OK=FALSE;
	    if (!strchr(p,'.') && !strchr(p,G_DIR_SEPARATOR) && !strchr(p,' ')) {
	      for (;*p;p++) {
		if (*p < '0' || *p > '9'){
		    OK=FALSE;
		    break;
		} else OK=TRUE;  
	      }
	    }
	    if (OK) return "application/x-trash";
	}
	
	if ((f=fopen(file,"rb"))==NULL) return "undetermined type";
	b=fread(buffer,1,256,f);
	fclose(f);
	for (i=0;i<b;i++) {
		if (buffer[i] < 6  || (buffer[i] > 14  && buffer[i] < 26)) 
			return "application/octet-stream";
	}
	return "text/plain";
}

static
const gchar *mime_magic_type(const gchar *file){
	GList *tmp;
	const char *result=NULL;
	unsigned char *t;
	size_t b;
	int matched_priority=0;
	int offset;

	const gchar *m=mimeable_file(file);
	if (!m) return "undetermined type";
	if (strlen(m)) return m;

	if (!magic_list) mime_build_magic();
	t=(unsigned char *)malloc(magic_max+1);

	for (tmp=magic_list;tmp; tmp=tmp->next){
		FILE *f;
		gboolean (*comp)(unsigned char *value,unsigned char *t,char *mask);
		
		magic_t *magic=(magic_t *)tmp->data;
		/*printf("examining: %s=%d {type=%s value=%s offset=%d mask=%s}\n",
				magic->mimetype,magic->priority, 
				magic->type,magic->value,magic->offset,
				magic->mask?magic->mask:"");*/
		if (!magic->priority){
		   /* printf("!!magic->priority\n");*/
		       continue;
		}
		if (magic->priority <= matched_priority) continue;
		for (offset=magic->offset; offset <= magic->last_offset; offset++){
			int bytes=0;
			if ((f=fopen(file,"rb"))==NULL) continue;
			if (fseek(f,offset,SEEK_SET)<0){
				fclose(f);
				continue;
			}
			if (strcmp(magic->type,"string")==0) {
				int l=0;
				char *p;
				for (p=magic->value; *p; p++) if (*p=='\\') l++;
				bytes = strlen(magic->value) - 3*l;
				if (bytes <= 0) continue;
				if  (bytes > magic_max) {
					g_warning("%s: bytes(%zd) > magic_max (%d), priority=%d, value=%s",
					magic->mimetype,bytes,magic_max,
					magic->priority,magic->value);
					continue;
				}
				comp = string_compare;
			} else if (strcmp(magic->type,"big16")==0 || strcmp(magic->type,"little16")==0){
				bytes = 2;
			       	comp = int16_compare;
			} else if (strcmp(magic->type,"big32")==0 || strcmp(magic->type,"little32")==0){
				bytes = 4;
			       	comp = int32_compare;
			} else if (strcmp(magic->type,"host32")==0){
				bytes = 4;
			      	comp = host32_compare;
			} else if (strcmp(magic->type,"host16")==0){
				bytes = 2;
			      	comp = host16_compare;
			} else if (strcmp(magic->type,"byte")==0) {
				bytes = 1;
				comp = byte_compare;
			} else continue;
			
			if (bytes > magic_max){
		       	       g_assert_not_reached();
			}

			b=fread(t,1,bytes,f);
			fclose(f);
			if (b < bytes) continue; /* short read */
			
			if ((*comp)(magic->value,t,magic->mask)){
				D(printf("***********************************match on %s\n",magic->mimetype);)
				matched_priority = magic->priority;
				result=(const char *)magic->mimetype;
				break;
			}
		}				
	}
	g_free(t);
	if (!result) return last_type_determination(file);
	return result;
}

G_MODULE_EXPORT
const gchar *mime_typeinfo(const gchar *type){
 xmlNodePtr node;
 xmlNodePtr subnode;
 xmlDocPtr doc;
 gchar *mimefile=NULL,*mimetype=NULL;
 static gchar *info;
	 
   	mimefile=g_build_filename(FREEDESKTOP_MIME_FILE,NULL);
    
#ifdef DEBUG
            printf("DBG: mimefile=%s\n",mimefile);
#endif
    if (access(mimefile,R_OK)!=0){
	g_free(mimefile); mimefile=NULL;
#ifdef DEBUG
            printf("DBG: access(mimefile,R_OK)!=0\n");
#endif
	return NULL;
    }
    xmlKeepBlanksDefault(0);

    if((doc = xmlParseFile(mimefile)) == NULL)
    {
#ifdef DEBUG
      	printf("xfce4-modules: invalid xml file: %s\n",mimefile);
#endif
	g_free(mimefile); mimefile=NULL;
	return NULL;
    }
	
    node = xmlDocGetRootElement(doc);
    if(!xmlStrEqual(node->name, (const xmlChar *)"mime-info"))
    {
#ifdef DEBUG
      	printf("xfce4-modules: invalid xml file %s\n",mimefile);
#endif
	g_free(mimefile); mimefile=NULL;
        xmlFreeDoc(doc);
	return NULL;
    }
    
    /* Now parse the xml tree */
#ifdef DEBUG
            printf("DBG: parsing %s\n",mimefile);
	    printf("------------looking for %s\n",type);
#endif
    for(node = node->children; node; node = node->next)
    {
	if(xmlStrEqual(node->name, (const xmlChar *)"mime-type")){
	    mimetype = (char *)xmlGetProp(node, (const xmlChar *)"type");
	    D(printf("%s==%s\n",mimetype,type);)
	    if (xmlStrEqual(mimetype, (const xmlChar *)type)){
    		for(subnode = node->children; subnode; subnode = subnode->next){
		   if(xmlStrEqual(subnode->name, (const xmlChar *)"comment")){
	    		char *e=xmlNodeListGetString(doc, subnode->children, 1);
			g_free(mimetype);
	    		xmlFreeDoc(doc);
    			g_free(mimefile);
			if (info) g_free(info);
			info =g_strdup(e);
			if (e) g_free(e);
			return (const gchar *)info;	
		   }
		}
		    
	    }
	    g_free(mimetype);
	}
    }
    xmlFreeDoc(doc);
    g_free(mimefile);
    return NULL;
}

static  mime_t *
locate_mime_t(const gchar *file){
	mime_t *mime;
	const gchar *p;
	if (!application_hash) mime_build_list();
	if (strrchr(file,G_DIR_SEPARATOR)) p=strrchr(file,G_DIR_SEPARATOR)+1;
	else p=file;
	do {
	   gchar *g;
	   if (*p =='.') p++;
	   /* try all lower case (hash table keys are set this way)*/
	   g=g_utf8_strdown (p,-1);
	   D(printf("lOOking for %s\n",g);)
	   mime = g_hash_table_lookup(application_hash, g);
	   g_free(g);
	   if (mime) {
	       D(printf("OK\n");)
	       return mime;
	   } 
	   D(else  printf("NOT FOUND\n");)
	} while ((p=strchr(p,'.'))!=NULL);
	return NULL;
}


G_MODULE_EXPORT
const gchar *mime_get_type(const gchar *file,gboolean try_magic){
	mime_t *mime;
	const gchar *m=mimeable_file(file);

	if (m && strlen(m)) return m;  
  	
	mime=locate_mime_t(file);
	if (mime) return (const gchar *)(mime->mimetype);

	if (!m) return "undetermined type";
	/* try magic */
	if (try_magic) return mime_magic_type(file);
	else return last_type_determination(file);	
}

void g_module_unload(GModule *module){
	 GList *tmp;
	if (application_hash){
	   destroy_application_hash(application_hash);
	   application_hash=NULL;
	}
	if (magic_list){
	   for (tmp=magic_list;tmp; tmp=tmp->next){
		   magic_t *magic =(magic_t *)tmp->data;
		   if (magic->type) g_free(magic->type);
		   if (magic->value) g_free(magic->value);
		   if (magic->mask) g_free(magic->mask);
		   g_free(magic);
	   }
	   g_list_free(magic_list);
	   magic_list=NULL;
	}
	if (xfmime_fun) g_free(xfmime_fun);
	xfmime_fun = NULL;
}

#endif


