/*  xfce4
 *  Copyright (C) 2002 Jasper Huijsmans (huysmans@users.sourceforge.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>

#include <gdk/gdk.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkimage.h>

#include "xfce_iconbutton.h"

#define DEBUGGING 0

/* XXX: this is used by gtkbutton. hope they won't change it soon ... */
#define CHILD_SPACING 1

/* Forward declarations */

static void xfce_iconbutton_class_init (XfceIconbuttonClass * class);
static void xfce_iconbutton_init (XfceIconbutton * iconbutton);
static void xfce_iconbutton_finalize (GObject * object);

static void xfce_iconbutton_size_allocate (GtkWidget * widget,
					   GtkAllocation * allocation);

static gboolean xfce_iconbutton_scale (XfceIconbutton * iconbutton);

static void image_style_set (GtkWidget * widget, GtkStyle * old);

/* Local data */

static GtkWidgetClass *parent_class = NULL;

GtkType
xfce_iconbutton_get_type (void)
{
    static GtkType iconbutton_type = 0;

    if (!iconbutton_type)
    {
	static const GTypeInfo iconbutton_info = {
	    sizeof (XfceIconbuttonClass),
	    NULL,		/* base_init */
	    NULL,		/* base_finalize */
	    (GClassInitFunc) xfce_iconbutton_class_init,
	    NULL,		/* class_finalize */
	    NULL,		/* class_data */
	    sizeof (XfceIconbutton),
	    0,			/* n_preallocs */
	    (GInstanceInitFunc) xfce_iconbutton_init
	};

	iconbutton_type =
	    g_type_register_static (GTK_TYPE_BUTTON, "XfceIconbutton",
				    &iconbutton_info, 0);
    }

    return iconbutton_type;
}

static void
xfce_iconbutton_class_init (XfceIconbuttonClass * class)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (class);
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass *) class;
    widget_class = (GtkWidgetClass *) class;
    gobject_class = G_OBJECT_CLASS (class);

    parent_class = gtk_type_class (gtk_button_get_type ());

    widget_class->size_allocate = xfce_iconbutton_size_allocate;

    gobject_class->finalize = xfce_iconbutton_finalize;
}

static void
xfce_iconbutton_init (XfceIconbutton * iconbutton)
{
    iconbutton->image = gtk_image_new ();
    gtk_widget_show (iconbutton->image);
    gtk_container_add (GTK_CONTAINER (iconbutton), iconbutton->image);

    iconbutton->icon_width = iconbutton->icon_height = -1;
    iconbutton->pb = NULL;

    /* force style thickness programmatically */
    g_signal_connect (iconbutton->image, "style-set",
		      G_CALLBACK (image_style_set), NULL);

}

GtkWidget *
xfce_iconbutton_new (void)
{
    return GTK_WIDGET (g_object_new (xfce_iconbutton_get_type (), NULL));
}

GtkWidget *
xfce_iconbutton_new_from_pixbuf (GdkPixbuf * pixbuf)
{
    GtkWidget *iconbutton;

    iconbutton = xfce_iconbutton_new ();
    xfce_iconbutton_set_pixbuf (XFCE_ICONBUTTON (iconbutton), pixbuf);

    return iconbutton;
}

void
xfce_iconbutton_set_pixbuf (XfceIconbutton * iconbutton, GdkPixbuf * pixbuf)
{
    g_return_if_fail (iconbutton != NULL);
    g_return_if_fail (XFCE_IS_ICONBUTTON (iconbutton));

    if (iconbutton->pb)
	g_object_unref (iconbutton->pb);

    iconbutton->pb = pixbuf;
    g_object_ref (iconbutton->pb);

    if (iconbutton->icon_width > 0)
    {
	xfce_iconbutton_scale (iconbutton);
    }
    else
    {
	gtk_image_set_from_pixbuf (GTK_IMAGE (iconbutton->image),
				   iconbutton->pb);
    }
}

static void
xfce_iconbutton_finalize (GObject * object)
{
    XfceIconbutton *iconbutton;

    g_return_if_fail (object != NULL);
    g_return_if_fail (XFCE_IS_ICONBUTTON (object));

    iconbutton = XFCE_ICONBUTTON (object);

    if (iconbutton->pb)
    {
	g_object_unref (iconbutton->pb);
	iconbutton->pb = NULL;
    }

    G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
xfce_iconbutton_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
    XfceIconbutton *iconbutton;
    gint xthick, ythick, borderwidth;
    GtkBorder *tmp_border;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (XFCE_IS_ICONBUTTON (widget));
    g_return_if_fail (allocation != NULL);

    iconbutton = XFCE_ICONBUTTON (widget);

    borderwidth = gtk_container_get_border_width (GTK_CONTAINER (widget));

    xthick = 2 * (CHILD_SPACING + borderwidth + widget->style->xthickness +
		  iconbutton->image->style->xthickness);
    ythick = 2 * (CHILD_SPACING + borderwidth + widget->style->ythickness +
		  iconbutton->image->style->ythickness);

    gtk_widget_style_get (widget, "default_border", &tmp_border, NULL);

    if (tmp_border && GTK_WIDGET_CAN_DEFAULT (widget))
    {
	xthick += tmp_border->left + tmp_border->right;
	ythick += tmp_border->top + tmp_border->bottom;
	gtk_widget_queue_draw(widget);
    }

    if (iconbutton->icon_width == allocation->width - xthick &&
	iconbutton->icon_height == allocation->height - ythick)
    {
	parent_class->size_allocate (widget, allocation);
	return;
    }
    
    iconbutton->icon_width = allocation->width - xthick;
    iconbutton->icon_height = allocation->height - ythick;

    widget->allocation = *allocation;

#if DEBUGGING
    g_print (" ** size: %d x %d\n",
	     iconbutton->icon_width,
	     iconbutton->icon_height);
#endif
    xfce_iconbutton_scale (iconbutton);

    parent_class->size_allocate (widget, allocation);

#if DEBUGGING
    g_print (" ** allocated size: %d x %d\n",
	     iconbutton->image->allocation.width,
	     iconbutton->image->allocation.height);
#endif
}

/* returns TRUE when image is set, FALSE otherwise */
static gboolean
xfce_iconbutton_scale (XfceIconbutton * iconbutton)
{
    GdkPixbuf *tmp;
    double wratio, hratio;
    int w, h, pw, ph;

    g_return_val_if_fail (iconbutton != NULL, FALSE);
    g_return_val_if_fail (XFCE_IS_ICONBUTTON (iconbutton), FALSE);
    g_return_val_if_fail (iconbutton->pb != NULL, FALSE);
    g_return_val_if_fail (GDK_IS_PIXBUF (iconbutton->pb), FALSE);

    pw = gdk_pixbuf_get_width (iconbutton->pb);
    ph = gdk_pixbuf_get_height (iconbutton->pb);

    w = iconbutton->icon_width <= 0 ? pw : iconbutton->icon_width;
    h = iconbutton->icon_height <= 0 ? ph : iconbutton->icon_height;

    if (w >= pw && h >= ph)
    {
#if DEBUGGING
	g_print (" ** don't scale: %d x %d\n", pw, ph);
#endif
	/* don't scale up -> ugly icons */ 
	tmp = iconbutton->pb;
	g_object_ref (tmp);
    }
    else
    {
	wratio = (double) pw / w;
	hratio = (double) ph / h;

	if (wratio > hratio)
	    h = rint ((double) ph / wratio);
	else if (hratio > wratio)
	    w = rint ((double) pw / hratio);

#if DEBUGGING
	g_print (" ** scale image: %d x %d\n", w, h);
#endif
	tmp = gdk_pixbuf_scale_simple (iconbutton->pb, w, h,
				       GDK_INTERP_BILINEAR);
    }

    gtk_image_set_from_pixbuf (GTK_IMAGE (iconbutton->image), tmp);
    g_object_unref (tmp);

    return TRUE;
}

static void
real_style_set (GtkWidget * widget, GtkStyle * old)
{
    GtkRcStyle *style;

    style = gtk_widget_get_modifier_style (widget);
    style->xthickness = style->ythickness = 0;
    gtk_widget_modify_style (widget, style);
}

static void
image_style_set (GtkWidget * widget, GtkStyle * old)
{
    static int recursive = 0;

    if (recursive > 0)
	return;

    recursive++;
    real_style_set (widget, old);
    recursive--;
}
