/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 * Copyright (c) 2003 Benedikt Meurer <benedikt.meurer@unix-ag.uni-siegen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_UCRED_H
#include <sys/ucred.h>
#endif
#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif

#include <errno.h>
#ifdef HAVE_FSTAB_H
#include <fstab.h>
#endif
#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_MNTENT_H
#include <mntent.h>
#endif
#ifdef HAVE_PATHS_H
#include <paths.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_FS_TYPES_H
#include <sys/fs_types.h>
#endif

#ifdef HAVE_SYS_MNTTAB_H
#include <sys/mnttab.h>
#endif

#ifdef HAVE_SYS_VFS_H
#include <sys/vfs.h>
#endif

#ifdef HAVE_SYS_VFSTAB_H
#include <sys/vfstab.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "add_node_contents.h"
#include "entry.h"
#include "gdir.h"
#include "misc.h"
#include "refresh.h"
#include "tubo.h"


static gdir_t fstat_gdir;
extern int stop;
static pid_t Gpid;
static GtkTreeView *fstab_treeview;

static gint watch_stop(gpointer data)
{
    tree_details_t *tree_details = get_tree_details(fstab_treeview);

    if(!tree_details->tubo_object) return FALSE;
    if(stop)
    {
	char m[32];
	sprintf(m, "%d\n", Gpid);
	print_diagnostics(fstab_treeview, "xf_WARNING_ICON", _("Subprocess aborted, pid="), m, NULL);
	/*cancelled = TRUE;*/
	TuboCancel(tree_details->tubo_object, NULL);
	if(Gpid)
	    kill(Gpid, SIGHUP);	/*hack! */
	stop = FALSE;
	tree_details->tubo_object = NULL;
	hide_stop(tree_details->window);
	Gpid = 0;
	cursor_reset(fstab_treeview);
	return FALSE;
    }
    set_progress(fstab_treeview, -1, -1);
    return TRUE;
}

static int operate_stderr(int n, void *data)
{
    char *line;
    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    print_diagnostics(fstab_treeview, "xf_ERROR_ICON", line, NULL);
    return TRUE;
}
static int operate_stdout(int n, void *data)
{
    char *line;
    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    print_diagnostics(fstab_treeview, "xf_WARNING_ICON", line, NULL);
    return TRUE;
}


static void fork_finished_function(pid_t pid)
{
    char m[32];
    tree_entry_t *en;
    int status;
    GtkTreeIter iter;
    tree_details_t *tree_details = get_tree_details(fstab_treeview);
    sprintf(m, "%d\n", pid);
    /*print_diagnostics(fstab_treeview, "xf_INFO_ICON", _("Subprocess done, pid="), m, NULL);*/

    tree_details->tubo_object = NULL;
    hide_stop(tree_details->window);
    cursor_reset(fstab_treeview);
    /* another wait just in case the one in tubo misses 
     * (shouldn't happen, but it does)*/
    
    waitpid(pid, &status, WNOHANG);
    en= get_selected_entry(fstab_treeview, &iter);
    status=is_mounted(en->path);
    if (status < 0) TOGGLE_MOUNTED(en->type);
    else {
	    if (status) SET_MOUNTED(en->type);
	    else UNSET_MOUNTED(en->type);
    }
    if (IS_MOUNTED(en->type))
	on_refresh((GtkButton *)(tree_details->window),NULL);
    else {
    	GtkTreePath *treepath;
    	GtkTreeModel *treemodel = gtk_tree_view_get_model(fstab_treeview);
    	treepath = gtk_tree_model_get_path(treemodel, &iter);
    	gtk_tree_view_collapse_row(fstab_treeview, treepath);
    	remove_folder(fstab_treeview, &iter);
    	gtk_tree_path_free(treepath);
	update_icon(fstab_treeview,&iter);
    }
    
#ifdef DEBUG  
    printf("DBG:fstab over by fork_finished_function\n");
#endif
}

/* NOTE: linux will fail if HAVE_STATFS is before HAVE_GETMNTENT section */
int is_mounted(char *mnt_point){
	
#if defined(HAVE_GETMNTENT) && defined(HAVE_STRUCT_MNTENT_MNT_DIR)
	struct mntent *m;
	const gchar *mnttab;
	FILE *mtab_fd;
	FILE *tab_file;
#if defined(MNT_MNTTAB)
	mnttab = MNT_MNTTAB;
#else
	/*
	 * try /proc/mounts first
	 */
	if ((mtab_fd = fopen("/proc/mounts", "r")) != NULL) {
		(void)fclose(mtab_fd);
		mnttab = "/proc/mounts";
	}
	else
		mnttab = "/etc/mtab";
#endif
	tab_file=setmntent(mnttab, "r");

	if (!tab_file) {
	      perror("setmntent:");
	      return -1;
	}
	while ((m=getmntent(tab_file))!=NULL){
		/*printf("DBG2:%s=%s\n",m->mnt_dir,mnt_point);*/
	   if (strcmp(m->mnt_dir,mnt_point)==0){
	        endmntent(tab_file);
		return 1;
	   }	   
	}
	endmntent(tab_file);
	return 0; 
#elif defined(HAVE_STATFS) && defined(HAVE_STRUCT_STATFS_F_MNTONNAME)
      struct statfs statfsbuf;	
      if (statfs(mnt_point, &statfsbuf) == 0) {
	if (strcmp(mnt_point, statfsbuf.f_mntonname) == 0) return 1;
	else return 0;
      }
      return -1; 
#elif defined(HAVE_GETMNTENT) && defined(HAVE_STRUCT_MNTTAB_MNT_MOUNTP)
	/*
	 * The Solaris way...
	 */
	struct mnttab mbuf;
	FILE *fp;
	int r=-1;

	if ((fp = fopen("/etc/mnttab", "r")) != NULL) {
		for (;;) {
			if (getmntent(fp, &mbuf) != 0){
				r=0;
				break;
			}

			if (strcmp(mbuf.mnt_mountp,mnt_point) == 0) {
				r=1;
				break;
			}
		}
		(void)fclose(fp);
	}
	return r;
#else
    return -1;
#warning "missing code here. " 
#endif
}

int
fstab_mount(GtkTreeView *treeview, char *mnt_point, gboolean umnt)
{
    gchar *argument[5];
    gchar **ap;
    int i;
    tree_details_t *tree_details = get_tree_details(treeview);

    ap = argument;

    i=is_mounted(mnt_point);
    if (i >= 0){
      if (i) *ap++ = "umount";
      else *ap++ ="mount";
    } else {
      if (umnt)	*ap++ = "umount";
      else	*ap++ = "mount";
    }

#if !defined(HAVE_GETVFSENT) && !defined(STAT_STATVFS) 
    /* XXX - This test is BAD */
    /* -v is different on Solaris */
    *ap++ = "-v";
#endif

    *ap++ = mnt_point;
    *ap++ = NULL;

    print_status(treeview,"xf_INFO_ICON",_("Mount:"), mnt_point, NULL);
    fstab_treeview = treeview;
    
    chdir ("/");
    show_stop(tree_details->window);

    tree_details->tubo_object = 
		Tubo(fork_function, 
				(void *)argument, 
				fork_finished_function, 
				FALSE, 
				operate_stdout, 
				operate_stderr);
    	
    g_timeout_add_full(0,260, 
			(GtkFunction) watch_stop, 
			(gpointer) treeview,
			NULL);
    	
    Gpid = TuboPID(tree_details->tubo_object);
    return(1);
}

#if 0 /* NOTYET */
/*
 * Check if file system is already mounted
 */
static gboolean
is_mounted(const gchar *mountp)
{
#if defined(HAVE_STATFS) && defined(HAVE_STRUCT_STATFS_F_MNTONNAME)
	/*
	 * The BSD way...
	 */
	struct statfs statfsbuf;

	if (statfs(mountp, &statfsbuf) == 0) {
		if (strcmp(statfsbuf.f_mntonname, mountp) == 0)
			return(TRUE);
	}
#elif defined(HAVE_GETMNTENT) && defined(HAVE_STRUCT_MNTTAB_MNT_MOUNTP)
	/*
	 * The Solaris way...
	 */
	struct mnttab mbuf;
	FILE *fp;

	if ((fp = fopen("/etc/mnttab", "r")) != NULL) {
		for (;;) {
			if (getmntent(fp, &mbuf) != 0)
				break;

			if (strcmp(mbuf.mnt_mountp, mountp) == 0) {
				(void)fclose(fp);
				return(TRUE);
			}
		}
	}

	(void)fclose(fp);
#elif defined(HAVE_GETMNTENT) && defined(HAVE_STRUCT_MNTENT_MNT_DIR)
	/*
	 * The Linux way...
	 */
	struct mntent *mptr;
	FILE *fp;

	if ((fp = setmntent("/proc/mounts", "r")) == NULL)
		if ((fp = setmntent("/etc/mtab", "r")) == NULL)
			return(FALSE);

	for (mptr = getmntent(fp); mptr != NULL; mptr = getmntent(fp)) {
		if (strcmp(mptr->mnt_dir, mountp) == 0) {
			(void)endmntent(fp);
			return(TRUE);
		}
	}

	(void)endmntent(fp);
#endif

	return(FALSE);
}

static void
read_fstab(...)
{
}
#endif

/* NOTE: FreeBSD will fail if is HAVE_GETVFSENT before HAVE_STATFS */

#if defined(HAVE_GETMNTENT) && defined(HAVE_STRUCT_MNTENT_MNT_DIR)

/*
 * HP-UX, Linux, SunOS 4.x
 */

#if !defined(MNTTYPE_PROCFS)
#define MNTTYPE_PROCFS	"proc"
#endif

#if !defined(MNTTYPE_SMBFS)
#define MNTTYPE_SMBFS	"smbfs"
#endif

#if !defined(MNTTYPE_DEV)
#define MNTTYPE_DEV	"devpts"
#endif

#if !defined(MNTTYPE_SHM)
#define MNTTYPE_SHM	"tmpfs"
#endif

#if !defined(MNTTYPE_CDFS)
#define	MNTTYPE_CDFS	"hfs"		/* HP-UX */
#endif

#if !defined(MNTTYPE_SWAP)
#define MNTTYPE_SWAP	"swap"
#endif

#if !defined(MNTTYPE_SWAPFS)
#define	MNTTYPE_SWAPFS	"swapfs"	/* HP-UX */
#endif

#if !defined(MNTTYPE_NFS)
#define MNTTYPE_NFS	"nfs"
#endif

#if !defined(MNT_CHECKLIST)
#if defined(_PATH_FSTAB)
#define MNT_CHECKLIST	_PATH_FSTAB
#else
#define MNT_CHECKLIST	"/etc/fstab"
#endif
#endif

int
open_fstab(GtkTreeView *treeview, GtkTreeIter *iter, GtkTreePath *treepath, 
		gpointer user_data)
{
	/*struct mntent *mnt_mounted;*/
        struct mntent *mnt_struct;
   	tree_entry_t *en;
	const gchar *mnttab;
	/*struct stat st;*/
	FILE *fstab_fd;
	FILE *mtab_fd;
	int i;

#if defined(MNT_MNTTAB)
	mnttab = MNT_MNTTAB;
#else
	/*
	 * try /proc/mounts first
	 */
	if ((mtab_fd = fopen("/proc/mounts", "r")) != NULL) {
		(void)fclose(mtab_fd);
		mnttab = "/proc/mounts";
	}
	else
		mnttab = "/etc/mtab";
#endif
	
	if ((fstab_fd = setmntent(MNT_CHECKLIST,"r")) == NULL) {
		fprintf(stderr, "Unable to open %s\n", MNT_CHECKLIST);
		return(0);
	}

	fstat_gdir.pathc = 0;
	
	for (mnt_struct = getmntent(fstab_fd); mnt_struct != NULL;
			mnt_struct = getmntent(fstab_fd)) {
		if (strcmp(MNTTYPE_SWAP, mnt_struct->mnt_type) == 0)
			continue;

#if defined(MNTTYPE_IGNORE)
		if (strcmp(MNTTYPE_IGNORE, mnt_struct->mnt_type) == 0)
			continue;
#endif

		if (!g_file_test(mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
			continue;

		fstat_gdir.pathc++;
        }

        (void)endmntent(fstab_fd);
	
	en = get_entry(treeview, iter);

	fstat_gdir.gl = (dir_t *)malloc(fstat_gdir.pathc * sizeof(dir_t));

        if ((fstab_fd = setmntent(MNT_CHECKLIST,"r")) == NULL) {
		fprintf(stderr, "Unable to open %s\n", MNT_CHECKLIST);
		g_free(fstat_gdir.gl);
		return(0);
	}

	for (i = 0, mnt_struct = getmntent(fstab_fd); mnt_struct != NULL;
			mnt_struct = getmntent(fstab_fd)) {
		if (strcmp(MNTTYPE_SWAP, mnt_struct->mnt_type) == 0)
			continue;

#if defined(MNTTYPE_IGNORE)
		if (strcmp(MNTTYPE_IGNORE, mnt_struct->mnt_type) == 0)
			continue;
#endif

		if (!g_file_test(mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
			continue;

    		fstat_gdir.gl[i].en = stat_entry(mnt_struct->mnt_dir, en->type);
   	 	fstat_gdir.gl[i].pathv = g_strdup(mnt_struct->mnt_dir);

		SET_XF_FSTAB(fstat_gdir.gl[i].en->type);

		if (strcmp(MNTTYPE_NFS, mnt_struct->mnt_type) == 0)
			SET_NFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_SMBFS, mnt_struct->mnt_type) == 0)
			SET_NFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_PROCFS, mnt_struct->mnt_type) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_DEV, mnt_struct->mnt_type) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_SHM, mnt_struct->mnt_type) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_SWAPFS, mnt_struct->mnt_type) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_CDFS, mnt_struct->mnt_type) == 0)
			SET_CDFS_TYPE(fstat_gdir.gl[i].en->subtype);

		i++;

	}

        (void)endmntent(fstab_fd);

#if 0
/* MOUNTED flag not needed in this case */	
	for (i = 0; i < fstat_gdir.pathc; i++)
	{
		/*
		 * Check if filesystem is already mounted
		 */
		if ((mtab_fd = setmntent(mnttab, "r")) == NULL) {
			fprintf(stderr, "Unable to open %s\n", mnttab);
			continue;
		}

		for (mnt_mounted = getmntent(mtab_fd); mnt_mounted != NULL;
				mnt_mounted = getmntent(mtab_fd)) {
			if (!strcmp(fstat_gdir.gl[i].pathv,
				    mnt_mounted->mnt_dir)) {
				printf("%s: already mounted\n",
						mnt_mounted->mnt_dir);
		    		TOGGLE_MOUNTED(fstat_gdir.gl[i].en->type);
			}
		}

		(void)endmntent(mtab_fd);
	}
#endif

	add_node_contents(treeview, iter, &fstat_gdir);

	gdirfree(&fstat_gdir);

        return(1);
}

/* needs STAT_STATVFS to avert FreeBSD from falling in here,
 * but that does not work with fbsd 5.0, so use HAVE_SYS_STATFS_H
 * instead... */
#elif defined(HAVE_GETVFSENT) && defined (HAVE_SYS_STATFS_H)

/*
 * Solaris
 */

#define MNTTYPE_CACHEFS	"cachefs"
#define MNTTYPE_HSFS	"hsfs"
#define MNTTYPE_NFS	"nfs"
#define MNTTYPE_SWAP	"swap"

int
open_fstab(GtkTreeView *treeview, GtkTreeIter *iter, GtkTreePath *treepath, 
		gpointer user_data)
{
	struct vfstab vbuf;
	struct mnttab mbuf;
   	tree_entry_t *en;
	FILE *vfstab_fp;
	FILE *mnttab_fp;
	struct stat st;
	int i;

	if ((vfstab_fp = fopen("/etc/vfstab", "r")) == NULL)
		return(0);

	for (i = 0;; ) {
		if (getvfsent(vfstab_fp, &vbuf) != 0)
			break;
		
		if(strcmp(MNTTYPE_SWAP, vbuf.vfs_fstype) == 0)
			continue;

		if (!g_file_test(vbuf.vfs_mountp, G_FILE_TEST_IS_DIR))
			continue;

		i++;
	}

	if ((mnttab_fp = fopen("/etc/mnttab", "r")) == NULL) {
		(void)fclose(vfstab_fp);
		return(0);
	}

	(void)rewind(vfstab_fp);

	if ((fstat_gdir.gl = (dir_t *)malloc(i * sizeof(dir_t))) == NULL) {
		(void)fclose(mnttab_fp);
		(void)fclose(vfstab_fp);
		return(0);
	}

	fstat_gdir.pathc = i;

	en = get_entry(treeview, iter);

	for (i = 0;;) {
		if (getvfsent(vfstab_fp, &vbuf) != 0)
			break;
		
		if(strcmp(MNTTYPE_SWAP, vbuf.vfs_fstype) == 0)
			continue;

		if (!g_file_test(vbuf.vfs_mountp, G_FILE_TEST_IS_DIR))
			continue;

    		fstat_gdir.gl[i].en = stat_entry(vbuf.vfs_mountp, en->type);
   	 	fstat_gdir.gl[i].pathv = g_strdup(vbuf.vfs_mountp);

		SET_XF_FSTAB(fstat_gdir.gl[i].en->type);
	
		if (strcmp(MNTTYPE_NFS, vbuf.vfs_fstype) == 0)
			SET_NFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_HSFS, vbuf.vfs_fstype) == 0)
			SET_CDFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_CACHEFS, vbuf.vfs_fstype) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);

		i++;
		/*
		 * Check if filesystem is already mounted
		 *  
		 */
		(void)rewind(mnttab_fp);
		for (;;) {
			if (getmntent(mnttab_fp, &mbuf) != 0)
				break;

			if (strcmp(mbuf.mnt_mountp, vbuf.vfs_mountp) == 0) 
		    		TOGGLE_MOUNTED(fstat_gdir.gl[i].en->type);
		}
	}

        (void)fclose(mnttab_fp);
        (void)fclose(vfstab_fp);
	
	add_node_contents(treeview, iter, &fstat_gdir);

	gdirfree(&fstat_gdir);

        return(1);
}

#elif defined(HAVE_GETFSENT)

/*
 * AIX, BSD, OSF1, ULTRIX
 */

#if defined(MOUNT_CD9660)
#define MNTTYPE_CDFS	MOUNT_CD9660
#else
#define MNTTYPE_CDFS	"cdrfs"		/* AIX */
#endif

#if defined(MOUNT_CFS)
#define MNTTYPE_CODAFS	MOUNT_CFS
#elif defined(MOUNT_CODA)
#define MNTTYPE_CODAFS	MOUNT_CODA
#else
#define MNTTYPE_CODAFS	"coda"
#endif

#if defined(MOUNT_KERNFS)
#define MNTTYPE_KERNFS	MOUNT_KERNFS
#else
#define MNTTYPE_KERNFS	"kernfs"
#endif

#if defined(MOUNT_MFS)
#define MNTTYPE_MFS	MOUNT_MFS
#else
#define MNTTYPE_MFS	"mfs"
#endif

#if defined(MOUNT_NFS)
#define MNTTYPE_NFS	MOUNT_NFS
#else
#define MNTTYPE_NFS	"nfs"
#endif

#if defined(MOUNT_PROCFS)
#define MNTTYPE_PROCFS	MOUNT_PROCFS
#else
#define MNTTYPE_PROCFS "procfs"
#endif

#if defined(MOUNT_SMBFS)
#define MNTTYPE_SMBFS	MOUNT_SMBFS
#else
#define MNTTYPE_SMBFS	"smbfs"
#endif

#define MNTTYPE_SWAP	"swap"

int
open_fstab(GtkTreeView *treeview, GtkTreeIter *iter, GtkTreePath *treepath, 
	   gpointer user_data)
{
#if defined(HAVE_STATFS) && defined(HAVE_STRUCT_STATFS_F_MNTONNAME)
	struct statfs statfsbuf;
#endif
	struct fstab *fs;
   	tree_entry_t *en;
	int i;

	en = get_entry(treeview, iter);

	if (!setfsent())
		return(0);

	for (i = 0, fs = getfsent(); fs != NULL;
			fs = getfsent()) {
		if (strcmp(MNTTYPE_SWAP, fs->fs_vfstype) == 0)
			continue;

		if (!g_file_test(fs->fs_file, G_FILE_TEST_IS_DIR))
			continue;

		i++;
	}

	if (!setfsent())
		return(0);
	
	if ((fstat_gdir.gl = (dir_t *)malloc(i * sizeof(dir_t))) == NULL) {
		endfsent();
		return(0);
	}

	fstat_gdir.pathc = i;

	for (i = 0, fs = getfsent(); fs != NULL; 
			fs = getfsent()) {
		if (strcmp(MNTTYPE_SWAP, fs->fs_vfstype) == 0)
			continue;

		if (!g_file_test(fs->fs_file, G_FILE_TEST_IS_DIR))
			continue;

    		fstat_gdir.gl[i].en = stat_entry(fs->fs_file, en->type);
   	 	fstat_gdir.gl[i].pathv = g_strdup(fs->fs_file);

		SET_XF_FSTAB(fstat_gdir.gl[i].en->type);

		if (strcmp(MNTTYPE_CDFS, fs->fs_vfstype) == 0)
			SET_CDFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_CODAFS, fs->fs_vfstype) == 0)
			SET_NFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_KERNFS, fs->fs_vfstype) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_MFS, fs->fs_vfstype) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_NFS, fs->fs_vfstype) == 0)
			SET_NFS_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_PROCFS, fs->fs_vfstype) == 0)
			SET_PROC_TYPE(fstat_gdir.gl[i].en->subtype);
		else if (strcmp(MNTTYPE_SMBFS, fs->fs_vfstype) == 0)
			SET_NFS_TYPE(fstat_gdir.gl[i].en->subtype);
#if 0
		/* no longer needed to set lag in this case */
#if defined(HAVE_STATFS) && defined(HAVE_STRUCT_STATFS_F_MNTONNAME)
		/*
		 * Check if file system is already mounted
		 */
		if (statfs(fs->fs_file, &statfsbuf) == 0) {
			if (strcmp(fs->fs_file, statfsbuf.f_mntonname) == 0) {
				printf("%s: already mounted\n", fs->fs_file);
		    		TOGGLE_MOUNTED(fstat_gdir.gl[i].en->type);
			}
		}
#endif
#endif

		i++;
	}

	endfsent();
	
	add_node_contents(treeview, iter, &fstat_gdir);

	gdirfree(&fstat_gdir);
		
        return(1);
}
#else

int
open_fstab(GtkTreeView *treeview, GtkTreeIter *iter, GtkTreePath *treepath, 
	   gpointer user_data)
{
	/* make GCC happy */
	(void)&fstat_gdir;

	return(1);
}

#endif

void
on_mount                          (GtkMenuItem     *menuitem,
                                        gpointer         user_data){
    GtkTreeIter iter;
    tree_entry_t *en;
    GtkTreeView *treeview = get_treeview((GtkWidget *) menuitem);
    en = get_selected_entry(treeview, &iter);
    if(!en || (!IS_FSTAB_TYPE(en->type) && !IS_ROOT_TYPE(en->type)))
	    return;
    fstab_mount(treeview,en->path,(int)((long)user_data));
}

