/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 * Some code in here contributed by the FSF
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif
#include <errno.h>
#ifdef HAVE_GRP_H
#include <grp.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_REGEX_H
#include <regex.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "glade_gui.h"
#include "glade_support.h"
#include "glade_callbacks.h"

#include "constants.h"
#include "types.h"

#include "dummies.h"
#include "entry.h"
#include "icons.h"
#include "ls.h"
#include "misc.h"

extern int stop;


/*
 * This function converts a time value specified in seconds since 1970-01-01
 * to a string representation. It shows the date and the time if the point
 * if less then six month in the past and only the date otherwise.
 * The exact format must be provided by a translation string. 
 *
 * The function should be thread-save since there are not used static
 * (or even global) variables if the system provided a localtime_r() function.
 *
 * Arguments:     when:    time in seconds since 1970-01-01
 *                string:  the result char-array in which the string is placed
 *                length:  the length of the string-array
 *                
 * Return value:  string on success, NULL on failure
 */
char *time_to_string (time_t when) {
      time_t          now             = time(NULL);
#ifdef HAVE_LOCALTIME_R
      struct tm       timestruct;
#endif
      struct tm*      timestruct_ptr;
      char*           formatstring;
      static char string[64];
      
#ifdef HAVE_MEMSET
      memset (string, 0, 64);
#else
      string[0]=0;
#endif
      
      formatstring = difftime(now, when) > 24*60*60*30*6
            /* strftime format for non-recent files (older than 6 months)  */
          ? _("%b %e  %Y")
            /* strftime format for recent files */
          : _("%b %e %H:%M");

#ifdef HAVE_LOCALTIME_R
      timestruct_ptr = &timestruct;
      localtime_r(&when, timestruct_ptr);
#else
      timestruct_ptr = localtime(&when);
#endif

      if (strftime(string, 64, formatstring, localtime(&when)) == 0) {
          return NULL;
      }

      return string;
}


char *sizetag(long long tama, int count)
{
    char *tag = "B";
    static char buf[64];

    buf[0] = 0;
    if(tama >= 0)
    {
	if(tama >= (unsigned long long)1024 * 1024 * 1024)
	{
	    (unsigned long long)tama /= (unsigned long long)1024 *1024 * 1024;
	    tag = "GB";
	}
	else if(tama >= 1024 * 1024)
	{
	    (unsigned long long)tama /= 1024 * 1024;
	    tag = "MB";
	}
	else if(tama >= 1024)
	{
	    (unsigned long long)tama /= 1024;
	    tag = "KB";
	}
	if(count > 0)
	    sprintf(buf, "%d %s, %llu %s.", count, (count == 1) ? _("file") : _("files"), tama, tag);
	else
	    sprintf(buf, "%llu %s", tama, tag);
    }
    else
    {
	if(count >= 0)
	    sprintf(buf, "%d %s", count, (count == 1) ? _("file") : _("files"));
	else
	    sprintf(buf, " ");
    }
    return buf;
}


int count_files(char *path)
{
    DIR *directory;
    struct dirent *dir;
    int count = 0;

    directory = opendir(path);
    if(!directory)
	return -1;
    while((dir = readdir(directory)) != NULL)
    {
	if(strcmp(dir->d_name, ".") && strcmp(dir->d_name, "..")
			/* not good: && strcmp(dir->d_name, "..Wastebasket")*/
			)		
		count++;
    }
    closedir(directory);
    return (count);
}




int count_hidden_files(char *path)
{
    DIR *directory;
    struct dirent *dir;
    int count = 0;

    directory = opendir(path);
    if(!directory)
	return -1;
    while((dir = readdir(directory)) != NULL)
    {
	if(strcmp(dir->d_name, ".") == 0 || 
	   strcmp(dir->d_name, "..") == 0 || 
	   strcmp(dir->d_name, "..Wastebasket") == 0)
	    continue;
	if(dir->d_name[0] == '.')
	{
	    count++;
	}
    }
    closedir(directory);
    return (count);
}

void update_columns(GtkTreeModel * treemodel, GtkTreeIter * target, tree_entry_t * en)
{
    struct group *g;
    struct passwd *p;
    char *group, *owner, *tag;
    char date[64];

    if(IS_DIR(en->type))
	tag = sizetag(-1, en->count);
    else if(IS_FILE(en->type) || IS_NETFILE(en->subtype))
	    	tag = sizetag((long long)en->st->st_size, -1);
    else
	tag = "";

    if((g = getgrgid(en->st->st_gid)) != NULL)
	group = g->gr_name;
    else if ((int)en->st->st_gid < 0) 
	group = "";
    else
	group = "?";
    if((p = getpwuid(en->st->st_uid)) != NULL)
	owner = p->pw_name;
    else if ((int)en->st->st_uid < 0)
	owner = "";
    else
	owner = "?";
    gtk_tree_store_set((GtkTreeStore *) treemodel, target, 
	MODE_COLUMN, mode_string(en->st->st_mode), 
	DATE_COLUMN, my_utf_string(time_to_string(en->st->st_mtime)), 
	GROUP_COLUMN, group, 
	OWNER_COLUMN, owner, 
	SIZE_COLUMN, tag, 
	-1);
}


static void add_file_info(GtkTreeView * treeview, GtkTreeIter * iterator, tree_entry_t * en, char *name)
{

    gchar *locale_name;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    tree_details_t *tree_details = get_tree_details(treeview);
    GError *error = NULL;

    /* this is to generate the translation strings */
    char *ignore_this_warning[] = {
	N_("Wastebasket")
    };

    /* to make GCC happy */
    (void)&ignore_this_warning;

    if(!tree_details->window)
	return;
    /* double dot files are not considered hidden in xffm */
    if(strncmp(name, "..", 2) == 0 && strlen(name) > 2)
	locale_name = _(name + 2);
    else
	locale_name = name;



    gtk_tree_store_set((GtkTreeStore *) treemodel, iterator, 
		    NAME_COLUMN, my_utf_string(locale_name), 
		    ENTRY_COLUMN, (gpointer) en, 
		    STYLE_COLUMN, PANGO_STYLE_NORMAL, 
		    SIZE_COLUMN, sizetag((long long)en->st->st_size, -1), -1);
    update_columns(treemodel, iterator, en);
    /* size info true on reloads */
    if(IS_DIR(en->type) || 
       IS_XF_NETWG(en->subtype) || 
       IS_XF_NETWS(en->subtype) ||
       IS_NETDIR(en->subtype) ||
       IS_XF_NETSHARE(en->subtype))
    {
	add_dummy(treeview, iterator);
	if(IS_LOCAL_TYPE(en->type))
	{
	    en->count = count_files(en->path);
	    gtk_tree_store_set((GtkTreeStore *) treemodel, iterator, SIZE_COLUMN, sizetag(-1, en->count), -1);
	}

    }
    set_icon(treeview, iterator);
    if(error)
	g_error_free(error);
    return;
}




GtkTreeIter add_it(GtkTreeView * treeview, GtkTreeIter * target, tree_entry_t * en, char *name, gboolean insert)
{
    GtkTreePath *treepath;
    GtkTreeIter iterator;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    tree_details_t *tree_details = get_tree_details(treeview);
    if(!tree_details->window)
	return iterator;
    if (insert)
	gtk_tree_store_prepend((GtkTreeStore *) treemodel, &iterator, target);
    else
	gtk_tree_store_append((GtkTreeStore *) treemodel, &iterator, target);
	      
    add_file_info(treeview, &iterator, en, name);
    treepath = gtk_tree_model_get_path(treemodel, &iterator);
    gtk_tree_path_free(treepath);
    return iterator;
}

GtkTreeIter add_file(GtkTreeView * treeview, GtkTreeIter * target, tree_entry_t * en, char *name)
{
    return add_it(treeview, target, en, name, FALSE);
}

GtkTreeIter prepend_file(GtkTreeView * treeview, GtkTreeIter * target, tree_entry_t * en, char *name)
{
    return add_it(treeview, target, en, name, TRUE);
}

