/*
 * radio plugin for Xfce4.
 *
 * Copyright (C) 2006-2011 Stefan Ott, All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Plugin layout
 *
 * ,--[ ebox ]--------------------------------------------------.
 * |,--[ outer_box ] ------------------------------------------.|
 * ||                                                          ||
 * || ,--[ box ] ----------------------. ,--[ signal_image ]-. ||
 * || |                                | |                   | ||
 * || | [label]     [signal_bar]       | |                   | ||
 * || |                                | |                   | ||
 * || '--------------------------------' '-------------------' ||
 * |'----------------------------------------------------------'|
 * '------------------------------------------------------------'
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include <sys/time.h>

#include "xfce4-radio.h"
#include "../icons/signal.xpm"

#include "radio.h"
#include "v4l2.h"

#include <libxfcegui4/dialogs.h>
#include <libxfcegui4/xfce-exec.h>
#include <libxfce4util/libxfce4util.h>
#include <gdk/gdkkeysyms.h>

#define SIGNAL_WIDTH 25
#define SIGNAL_HEIGHT 27

static GdkPixmap *signal_s;

static void
update_signal_image(radio_gui* data, gint signal)
{
	GdkImage *image;

	signal_s = gdk_pixmap_create_from_xpm_d
		(GTK_WIDGET(data->plugin)->window, NULL, NULL, signal_xpm);
	image = gdk_drawable_get_image(signal_s, SIGNAL_WIDTH * signal, 0,
		SIGNAL_WIDTH, SIGNAL_HEIGHT);

	gtk_image_set_from_image(GTK_IMAGE(data->signal_image), image, NULL);
	gtk_widget_show(data->signal_image);

	if (GTK_WIDGET(data->signal_image)->parent == NULL)
	{
		gtk_box_pack_start(GTK_BOX(data->outer_box),
			data->signal_image, FALSE, FALSE, 1);
	}
	g_object_unref(signal_s);
	g_object_unref(image);
}

static void
update_signal_bar(radio_gui* data, gint signal)
{
	GdkColor color;

	gtk_widget_show(data->signal_bar);
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
		(data->signal_bar), signal / data->max_signal_strength);

	if (signal == 1)
		gdk_color_parse(COLOR_SIGNAL_LOW, &color);
	else if (signal == data->max_signal_strength)
		gdk_color_parse(COLOR_SIGNAL_HIGH, &color);
	else
		gdk_color_parse(COLOR_SIGNAL_MED, &color);

	gtk_widget_modify_bg(data->signal_bar, GTK_STATE_PRELIGHT, &color);
}

static gboolean
update_signal(radio_gui* data)
{
	if (!data->on || !data->show_signal)
	{
		data->signal_timeout_id = 0;
		gtk_widget_hide(data->signal_bar);

		if (data->on && !data->show_signal)
			gtk_widget_hide(data->signal_image);

		if (!data->on && data->show_label_when_off)
			gtk_widget_hide(data->signal_image);

		return FALSE;
	}
	else
	{
		double signal = radio_get_signal();

		if (data->signal_timeout_id == 0)
		{
			data->signal_timeout_id = g_timeout_add(500,
				(GtkFunction) update_signal, (gpointer) data);
		}

		if (data->show_signal_graphical)
		{
			gtk_widget_hide(data->signal_bar);
			update_signal_image(data, signal);
		}
		else
		{
			gtk_widget_hide(data->signal_image);
			update_signal_bar(data, signal);
		}
		return TRUE;
	}
}

#define make_preset_finder(name, arg, search_arg, get_col, get_var,	\
						free, comparison)	\
	static gboolean							\
	find_preset_by_##name(GtkTreeModel *model,			\
			search_arg,					\
			GtkTreeIter *search_iter)			\
	{								\
		arg;							\
		GtkTreeIter iter;					\
		gboolean valid;						\
		valid = gtk_tree_model_get_iter_first(model, &iter);	\
		while (valid)						\
		{							\
			gtk_tree_model_get(model, &iter,		\
					get_col, get_var, -1);		\
			if (comparison) {				\
				*search_iter = iter;			\
				return TRUE;				\
			}						\
			free;						\
			valid = gtk_tree_model_iter_next(model, &iter);	\
		}							\
		return FALSE;						\
	}								\

make_preset_finder(freq, gint freq, gint search_freq, 1, &freq, freq = 0,
							freq == search_freq);
make_preset_finder(name, gchar *name, const gchar *search_name, 0, &name,
				g_free(name), strcmp(name, search_name) == 0);

static void
update_tooltip(radio_gui* data)
{
	GtkWidget* ebox = data->ebox;
	GtkTreeIter iter;
	gchar *name;

	DBG("Updating tooltip");

	GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
	if (find_preset_by_freq(presets, data->freq, &iter))
	{
		gtk_tree_model_get(presets, &iter, 0, &name, -1);
	}
	else
	{
		name = g_strdup_printf("%5.1f", (float) data->freq / 100);
	}

	gtk_tooltips_set_tip(data->tooltips, ebox, name, NULL);

	g_free(name);
}

static void
update_label_radio_on(radio_gui *data)
{
	gchar *label;

	if (data->show_label)
		gtk_widget_show(data->label);
	else
		gtk_widget_hide(data->label);

	if (data->resolve_presets_in_label)
	{
		GtkTreeIter iter;
		GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
		if (find_preset_by_freq(presets, data->freq, &iter))
			gtk_tree_model_get(presets, &iter, 0, &label, -1);
		else
			label = g_strdup_printf
				("%5.1f", ((float) data->freq) / 100);
	}
	else
		label = g_strdup_printf("%5.1f", ((float) data->freq) / 100);

	gtk_label_set_label(GTK_LABEL(data->label), label);
	g_free(label);

	update_tooltip(data);
}

static void
update_label_radio_off(radio_gui *data)
{
	gchar *label;
	label = g_strdup(_("- off -"));
	if (data->show_label_when_off)
	{
		gtk_widget_hide(data->signal_image);
		gtk_widget_show(data->label);
	}
	else
	{
		gtk_widget_hide(data->label);
		update_signal_image(data, 0);
	}
	gtk_label_set_label(GTK_LABEL(data->label), label);
	gtk_tooltips_set_tip(data->tooltips, data->ebox, label, NULL);

	g_free(label);
}

static void
update_label(radio_gui *data)
{
	if (data->on)
		update_label_radio_on(data);
	else
		update_label_radio_off(data);
}

static gboolean
update_radio(radio_gui *data)
{
	// We could just remove the timer, but then we would have to make sure
	// to re-add it at all the appropriate places, thus we don't.
	if (!data->auto_update_display)
		return TRUE;

	if (radio_is_muted())
	{
		if (data->on)
		{
			// Radio is off, we thought it was on
			if (data->show_signal)
				gtk_widget_hide(data->signal_bar);
			data->on = FALSE;
		}
	}
	else
	{
		if (!data->on)
		{
			// Radio is on, we thought it was off
			gtk_tooltips_enable(data->tooltips);
			data->on = TRUE;
		}
		data->freq = radio_get_freq() * 100;
		DBG("Updating frequency to %d", data->freq);
	}
	update_label(data);
	update_signal(data);

	return TRUE;
}

static void
xfce4_radio_tune(radio_gui* data)
{
	double freq = data->freq / 100.0;
	radio_set_freq(freq);
	DBG("Tuning to %f", freq);

	update_label(data);
	update_signal(data);

	write_config(data, FALSE);
}

static gboolean
xfce4_radio_start(radio_gui* data)
{
	DBG("Starting %s", data->device);

	if (!radio_init(data->device, DRIVER_V4L2))
	{
		DBG("Failed!");
		GtkWindow* win = GTK_WINDOW(gtk_widget_get_toplevel(
								data->box));
		GtkWidget* warn = gtk_message_dialog_new(win, 0,
			GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
			_("Error opening radio device"));
		gtk_dialog_run(GTK_DIALOG(warn));
		gtk_widget_destroy(warn);
		return FALSE;
	}

	if (strcmp(data->startup_command, "") != 0)
	{
		xfce_exec(data->startup_command, FALSE, FALSE, NULL);
	}
	xfce4_radio_tune(data);
	gtk_tooltips_enable(data->tooltips);
	return TRUE;
}

static void
xfce4_radio_stop(radio_gui* data)
{
	DBG("Stopping, auto_update_display=%d", data->auto_update_display);

	if (data->auto_update_display)
		radio_mute();
	else
	{
		radio_stop();
	}

	gtk_tooltips_disable(data->tooltips);

	if (data->show_signal) gtk_widget_hide(data->signal_bar);

	if (strcmp(data->shutdown_command, "") != 0)
	{
		xfce_exec(data->shutdown_command, FALSE, FALSE, NULL);
	}
}

static gint
parse_freq(const gchar *freq_char)
{
	gint mult, decimal_int;

	gint freq_int = 100 * atoi(freq_char);

	gchar *decimals = strstr(freq_char, ".");
	if (decimals)
		decimals++;
	else
	{
		// In some languages, a comma can be used instead of a point
		decimals = strstr(freq_char, ",");
		if (decimals)
			decimals++;
		else
			decimals = "0";
	}

	decimal_int = atoi(decimals);

	if (decimal_int < 10)
		mult = 10;
	else if (decimal_int < 100)
		mult = 1;
	else
		mult = 0;

	freq_int += mult * decimal_int;

	return freq_int;
}

static gboolean
parse_freq_and_tune(const char* freq_char, radio_gui* data)
{
	int freq_int = parse_freq(freq_char);
	DBG("Parsed frequency %d", freq_int);

	if (freq_int >= FREQ_MIN && freq_int <= FREQ_MAX)
	{
		data->freq = freq_int;
		xfce4_radio_tune(data);
		return TRUE;
	}
	return FALSE;
}

static void
radio_tune_gui(GtkEditable *menu_item, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	GtkWindow* win = GTK_WINDOW(gtk_widget_get_toplevel(data->box));
	GtkWidget* dialog = gtk_dialog_new_with_buttons(_("Tune radio"),
				NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);
	GtkWidget* box = GTK_DIALOG(dialog)->vbox;

	GtkWidget* label = gtk_label_new(_("Frequency [MHz]:"));
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

	GtkWidget* freq = gtk_entry_new_with_max_length(5);
	gtk_widget_show(freq);
	gtk_box_pack_start(GTK_BOX(box), freq, FALSE, FALSE, 0);

	int retval;
	for (;;)
	{
		retval = gtk_dialog_run(GTK_DIALOG(dialog));

		if (	retval == GTK_RESPONSE_CANCEL ||
			retval == GTK_RESPONSE_DELETE_EVENT ||
			retval == GTK_RESPONSE_NONE) {
				break;
		}

		const char* freq_char = gtk_entry_get_text(GTK_ENTRY(freq));
		if (parse_freq_and_tune(freq_char, data)) break;

		GtkWidget* warn = gtk_message_dialog_new(win, 0,
					GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						_("Illegal frequency."));
		gtk_dialog_run(GTK_DIALOG(warn));
		gtk_widget_destroy(warn);
	}
	gtk_widget_destroy(dialog);
}

static XfceRc *
get_config_rc()
{
	char *file;
	XfceRc *rc;

	// Take data from a clearly defined place RADIO_CONFIG
	file = xfce_resource_save_location(XFCE_RESOURCE_CONFIG, RADIO_CONFIG,
									TRUE);

	if (G_UNLIKELY(!file))
		return NULL;

	rc = xfce_rc_simple_open(file, FALSE);
	g_free(file);

	return rc;
}

static void
select_preset(GtkEditable* menu_item, radio_gui *data)
{
	GtkWidget* label = gtk_bin_get_child(GTK_BIN(menu_item));
	const gchar* text = gtk_label_get_text(GTK_LABEL(label));
	GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
	GtkTreeIter iter;

	if (find_preset_by_name(presets, text, &iter))
	{
		gtk_tree_model_get(presets, &iter, 1, &data->freq, -1);
		xfce4_radio_tune(data);
	}
}

static gboolean
mouse_click(GtkWidget* src, GdkEventButton *event, radio_gui* data)
{
	DBG("Mouse button %d clicked", event->button);
	if (event->button == 1)
	{
		if (!data->on)
			data->on = xfce4_radio_start(data);
		else
		{
			data->on = FALSE;
			xfce4_radio_stop(data);
		}
	}
	else if (event->button == 2 && data->on)
	{
		GtkWidget* menu = gtk_menu_new();
		GtkWidget* item = gtk_menu_item_new_with_label(_("Presets"));
		gtk_widget_show(item);
		gtk_menu_append(menu, item);
		gtk_widget_set_sensitive(item, FALSE);

		GtkWidget* separator = gtk_separator_menu_item_new();
		gtk_widget_show(separator);
		gtk_container_add(GTK_CONTAINER(menu), separator);
		gtk_widget_set_sensitive(separator, FALSE);

		gchar *name;
		GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);
		GtkTreeIter iter;

		gboolean valid = gtk_tree_model_get_iter_first(presets, &iter);
		while (valid)
		{
			gtk_tree_model_get(presets, &iter, 0, &name, -1);
			item = gtk_menu_item_new_with_label(name);
			gtk_widget_show(item);
			g_free(name);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
			g_signal_connect(GTK_WIDGET(item), "activate",
					G_CALLBACK(select_preset), data);
			valid = gtk_tree_model_iter_next(presets, &iter);
		}

		separator = gtk_separator_menu_item_new();
		gtk_widget_show(separator);
		gtk_container_add(GTK_CONTAINER(menu), separator);
		gtk_widget_set_sensitive(separator, FALSE);

		item = gtk_menu_item_new_with_label(_("Tune to frequency"));
		gtk_widget_show(item);
		gtk_menu_append(menu, item);
		g_signal_connect(GTK_WIDGET(item), "activate",
					G_CALLBACK(radio_tune_gui), data);

		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
				event->button, event->time);
	}
	update_label(data);
	update_signal(data);
	return event->button != 3;
}

static void
mouse_scroll(GtkWidget* src, GdkEventScroll *event, radio_gui* data)
{
	GtkTreePath *path;
	GtkTreeIter iter, iter1;
	GtkTreeModel *presets;

	DBG("Mouse scrolled");
	if (!data->on)
		return;

	int direction = event->direction == GDK_SCROLL_UP ? -1 : 1;
	if (data->scroll == CHANGE_FREQ) {
		data->freq += direction * FREQ_STEP;
		if (data->freq > FREQ_MAX) data->freq = FREQ_MIN;
		if (data->freq < FREQ_MIN) data->freq = FREQ_MAX;
		xfce4_radio_tune(data);
	}
	else if (data->scroll == CHANGE_PRESET)
	{
		presets = GTK_TREE_MODEL(data->presets);
		if (!find_preset_by_freq(presets, data->freq, &iter))
		{
			// tune to first preset, if it exists
			if (!gtk_tree_model_get_iter_first(presets, &iter))
				return;

			gtk_tree_model_get(presets, &iter,
					1, &data->freq, -1);
			xfce4_radio_tune(data);
			return;
		}
		// preset found
		if (direction == 1)
		{
			if (!gtk_tree_model_iter_next(presets, &iter))
				return;
			gtk_tree_model_get(presets, &iter, 1, &data->freq, -1);
			xfce4_radio_tune(data);
		}
		else
		{
			path = gtk_tree_model_get_path(presets, &iter);

			if (!gtk_tree_path_prev(path))
				return;

			gtk_tree_model_get_iter(presets, &iter1, path);
			gtk_tree_path_free(path);
			gtk_tree_model_get(presets, &iter1, 1, &data->freq, -1);
			xfce4_radio_tune(data);
		}
	}
}

static void
add_boxes(radio_gui *gui)
{
	if (gui->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		gui->box = gtk_hbox_new(FALSE, 0);
		gui->outer_box = gtk_hbox_new(FALSE, 0);
	}
	else
	{
		gui->box = gtk_vbox_new(FALSE, 0);
		gui->outer_box = gtk_vbox_new(FALSE, 0);
	}

	gtk_container_set_border_width(GTK_CONTAINER(gui->box), BORDER / 2);

	gtk_widget_show(gui->box);
	gtk_widget_show(gui->outer_box);

	gtk_container_add(GTK_CONTAINER(gui->outer_box), gui->box);
	gtk_container_add(GTK_CONTAINER(gui->ebox), gui->outer_box);
}

static void
add_signal_bar(radio_gui *gui)
{
	GtkWidget *signal_bar = gtk_progress_bar_new();

	if (gui->orientation == GTK_ORIENTATION_HORIZONTAL)
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR
			(signal_bar), GTK_PROGRESS_BOTTOM_TO_TOP);
	else
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR
			(signal_bar), GTK_PROGRESS_LEFT_TO_RIGHT);

	gtk_box_pack_start(GTK_BOX(gui->box), signal_bar, FALSE, FALSE, 1);
	gui->signal_bar = signal_bar;
}

static void
add_label(radio_gui *gui)
{
	gui->label = gtk_label_new("");
	gtk_widget_show(gui->label);

	gtk_box_pack_start(GTK_BOX(gui->box), gui->label, FALSE, FALSE, 1);
}

static void
add_image(radio_gui *gui)
{
	gui->signal_image = gtk_image_new();
}

static void
add_gui_elements(radio_gui *gui)
{
	add_boxes(gui);
	add_label(gui);
	add_signal_bar(gui);
	add_image(gui);
}

static radio_gui *
create_gui(gint orientation)
{
	radio_gui* gui;
	gui = g_new(radio_gui, 1);
	gui->max_signal_strength = 3;
	gui->orientation = orientation;

	gui->ebox = gtk_event_box_new();
	gtk_widget_show(gui->ebox);

	g_signal_connect(GTK_WIDGET(gui->ebox), "button_press_event",
						G_CALLBACK(mouse_click), gui);
	g_signal_connect(GTK_WIDGET(gui->ebox), "scroll_event",
						G_CALLBACK(mouse_scroll), gui);
	add_gui_elements(gui);

	return gui;
}

static void
free_presets(radio_gui *data)
{
	g_object_unref(data->presets);
}

static void
radio_free(XfcePanelPlugin *plugin, radio_gui *data)
{
	if (data->on) xfce4_radio_stop(data);
	free_presets(data);
	g_free(data);

	if (data->signal_timeout_id)
		g_source_remove(data->signal_timeout_id);
	if (data->radio_timeout_id)
		g_source_remove(data->radio_timeout_id);
}

static radio_gui *
plugin_control_new(XfcePanelPlugin *plugin)
{
	gint orientation = xfce_panel_plugin_get_orientation(plugin);
	radio_gui* plugin_data = create_gui(orientation);

	plugin_data->plugin = plugin;

	plugin_data->on = FALSE;
	plugin_data->freq = FREQ_INIT;
	strcpy(plugin_data->device, "/dev/radio0");

	plugin_data->show_signal = TRUE;
	plugin_data->show_signal_graphical = FALSE;
	plugin_data->show_label = TRUE;
	plugin_data->show_label_when_off = TRUE;
	plugin_data->auto_update_display = FALSE;
	plugin_data->presets = NULL;
	plugin_data->presets = NULL;
	plugin_data->scroll = CHANGE_FREQ;
	plugin_data->signal_timeout_id = 0;
	plugin_data->radio_timeout_id = g_timeout_add(2000,
			(GtkFunction)update_radio, (gpointer)plugin_data);
	plugin_data->tooltips = gtk_tooltips_new();
	plugin_data->startup_command[0] = '\0';
	plugin_data->shutdown_command[0] = '\0';

	update_label(plugin_data);

	gtk_container_add(GTK_CONTAINER(plugin), plugin_data->ebox);

	return plugin_data;
}

static void
radio_startup_command_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	const char* command = gtk_entry_get_text(GTK_ENTRY(editable));
	strncpy(data->startup_command, command, MAX_COMMAND_LENGTH);
}

static void
radio_shutdown_command_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	const char* command = gtk_entry_get_text(GTK_ENTRY(editable));
	strncpy(data->shutdown_command, command, MAX_COMMAND_LENGTH);
}

static void
radio_device_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	const char* device = gtk_entry_get_text(GTK_ENTRY(editable));
	strncpy(data->device, device, MAX_DEVICE_NAME_LENGTH);
}

static void
radio_show_signal_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	if (!data->show_label)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(editable),
			TRUE);
	data->show_signal = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
								(editable));
	update_signal(data);
}

static void
radio_show_label_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	if (!data->show_signal)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(editable),
			TRUE);
	data->show_label = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
								(editable));
	update_label(data);
}

#ifdef ENABLE_ANTENNA_ICON
static void
radio_show_label_when_off_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->show_label_when_off = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(editable));
	update_label(data);
}
#endif

static void
radio_resolve_presets_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->resolve_presets_in_label = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(editable));
	update_label(data);
}

#ifdef ENABLE_ANTENNA_ICON
static void
radio_signal_type_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->show_signal_graphical = !gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(editable));
	update_signal(data);

	DBG("show_signal_graphical = %d", data->show_signal_graphical);
}
#endif

static void
radio_auto_update_display_changed(GtkEditable* editable, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	data->auto_update_display = gtk_toggle_button_get_active
						(GTK_TOGGLE_BUTTON(editable));
}

static void
radio_scroll_type_changed(GtkEditable* button, void *pointer)
{
	radio_gui* data = (radio_gui*) pointer;
	gboolean frq = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	if (frq)
		data->scroll = CHANGE_FREQ;
	else
		data->scroll = CHANGE_PRESET;
}

static void
radio_plugin_dialog_response(GtkWidget *dialog, int response, radio_gui *data)
{
	gtk_widget_destroy(dialog);
	xfce_panel_plugin_unblock_menu(data->plugin);
	write_config(data, TRUE);
}

static void
cell_float_to_text_cb(GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *tree_model,
			GtkTreeIter *iter,
			gpointer data)
{
	gint f;
	gchar *text;

	// Get the double value from the model
	gtk_tree_model_get(tree_model, iter, 1, &f, -1);
	// Now we can format the value ourselves
	text = g_strdup_printf("%.2f", f / 100.0);
	g_object_set(cell, "text", text, NULL);
	g_free(text);
}

static void
add_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	gtk_tree_selection_get_selected(selection, &model, &iter);
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter,
				0, _("unnamed"),
				1, data->freq, -1);
	gtk_tree_selection_select_iter(selection, &iter);
	write_config(data, TRUE);
}

static void
del_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		if (gtk_list_store_remove(GTK_LIST_STORE(model), &iter))
		{
			gtk_tree_selection_select_iter(selection,&iter);
		}
	}
	write_config(data, TRUE);
	update_tooltip(data);
}

static void
up_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
		if (gtk_tree_path_prev(path))
		{
			GtkTreeIter iter1 = iter;
			gtk_tree_model_get_iter(model, &iter1, path);
			gtk_list_store_swap(GTK_LIST_STORE(model),
					&iter, &iter1);
		}
		gtk_tree_path_free(path);
	}
	write_config(data, TRUE);
	// update_tooltip shows the first preset with the currently tuned
	// frequency. If you have multiple entries with the same frequency,
	// moving them up or down changes which one is shown in the tooltip
	update_tooltip(data);
}

static void
down_button_clicked_cb(GtkWidget *widget, GtkTreeSelection *selection)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	radio_gui *data = g_object_get_data(G_OBJECT(selection), "data");

	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		GtkTreeIter iter1 = iter;
		if (gtk_tree_model_iter_next(model, &iter1))
		{
			gtk_list_store_swap(GTK_LIST_STORE(model),
					&iter, &iter1);
		}
	}
	write_config(data, TRUE);
	// See up_button_clicked_cb
	update_tooltip(data);
}

static gboolean
list_view_key_press_event_cb(GtkWidget *widget, GdkEventKey *event,
							GtkTreeSelection *data)
{
	if (event->keyval == GDK_Delete)
		del_button_clicked_cb(widget, data);
	if (event->keyval == GDK_Insert)
		add_button_clicked_cb(widget, data);

	return FALSE;
}

static void
name_cell_edited_cb(GtkCellRendererText *cellrenderertext, gchar *path_str,
						gchar *new_val, radio_gui *data)
{
	GtkTreeIter iter;
	gboolean valid;

	GtkListStore *presets = data->presets;
	valid = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(presets),
			&iter, path_str);
	g_return_if_fail(valid == TRUE);

	gtk_list_store_set(presets, &iter, 0, new_val, -1);
	write_config(data, TRUE);
	update_tooltip(data);
}

static void
freq_cell_edited_cb(GtkCellRendererText *cellrenderertext, gchar *path_str,
						gchar *new_val, radio_gui *data)
{
	GtkTreeIter iter;
	gboolean valid;
	gint value;

	value = parse_freq(new_val);

	if (value < FREQ_MIN) value = FREQ_MIN;
	if (value > FREQ_MAX) value = FREQ_MAX;

	GtkListStore *presets = data->presets;
	valid = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(presets),
			&iter, path_str);
	g_return_if_fail(valid == TRUE);

	gtk_list_store_set(presets, &iter, 1, value, -1);
	write_config(data, TRUE);
	update_tooltip(data);
}

static void
radio_plugin_create_options(XfcePanelPlugin *plugin, radio_gui *data)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *dialog;
	GtkWidget *frame;
	GtkWidget *real_frame;

	GtkWidget *show_signal;			// show the signal when on
	GtkWidget *show_label;			// show the label when on
#ifdef ENABLE_ANTENNA_ICON
	GSList *signal_type_group = NULL;	// show signal strength as:
	GtkWidget *signal_type_bar;		//  - progress bar
	GtkWidget *signal_type_xpm;		//  - graphical
	GSList *show_when_off = NULL;		// what to show when off:
	GtkWidget *label_when_off;		//  - label
	GtkWidget *graphics_when_off;		//  - graphics
#endif
	GSList *auto_update_display_group = NULL;// auto update display:
	GtkWidget *auto_update_display_yes;	//  - show
	GtkWidget *auto_update_display_no;	//  - hide
	GtkWidget *startup_command_entry;	// post-down command
	GtkWidget *shutdown_command_entry;	// post-down command
	GtkWidget *device_entry;		// v4l-device
	GSList *scroll_group = NULL;		// scroll action:
	GtkWidget *frequency_button;		//  - change frequency
	GtkWidget *preset_button;		//  - change preset
	GtkWidget *preset_box;
	GtkWidget *button_box;
	GtkWidget *notebook;
	GtkWidget *label_properties;
	GtkWidget *label_presets;
	GtkWidget *scrolled_window;
	GtkWidget *list_view;
	GtkTreeSelection *selection;
	GtkCellRenderer *cellrenderer;
	GtkTreeViewColumn *list_column;

	xfce_panel_plugin_block_menu(plugin);

	dialog = gtk_dialog_new_with_buttons(_("Properties"),
		GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
		GTK_DIALOG_DESTROY_WITH_PARENT |
		GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
		NULL);

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 2);

	g_signal_connect(dialog, "response",
			G_CALLBACK(radio_plugin_dialog_response), data);

	list_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(data->presets));
	gtk_widget_set_size_request(list_view, 200, 150);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list_view), FALSE);
	gtk_widget_show(list_view);

	cellrenderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(cellrenderer),
		"mode", GTK_CELL_RENDERER_MODE_EDITABLE,
		"editable", TRUE, NULL);
	list_column = gtk_tree_view_column_new_with_attributes
					(NULL, cellrenderer, "text", 0, NULL);
	gtk_tree_view_column_set_min_width(list_column, 300);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), list_column);
	g_signal_connect(GTK_OBJECT(cellrenderer), "edited",
			GTK_SIGNAL_FUNC(name_cell_edited_cb), (gpointer)data);

	cellrenderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(cellrenderer),
		"mode", GTK_CELL_RENDERER_MODE_EDITABLE,
		"editable", TRUE, \
		"xalign", 1.0f, NULL);
	list_column = gtk_tree_view_column_new_with_attributes
			(NULL, cellrenderer, "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(list_view), list_column);
	gtk_tree_view_column_set_cell_data_func(list_column, cellrenderer,
						cell_float_to_text_cb,
						NULL, NULL);
	g_signal_connect(GTK_OBJECT(cellrenderer), "edited",
			GTK_SIGNAL_FUNC(freq_cell_edited_cb), (gpointer)data);

	button_box = gtk_hbox_new(FALSE, 0);

	GtkWidget *add_button  = gtk_button_new_from_stock(GTK_STOCK_ADD);
	GtkWidget *del_button  = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	GtkWidget *up_button   = gtk_button_new_from_stock(GTK_STOCK_GO_UP);
	GtkWidget *down_button = gtk_button_new_from_stock(GTK_STOCK_GO_DOWN);

	gtk_box_pack_end(GTK_BOX(button_box), del_button, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_box), add_button, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_box), up_button,  FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_box), down_button,FALSE, FALSE, 0);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(list_view));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

	g_object_set_data(G_OBJECT(selection), "data", data);

	g_signal_connect(GTK_OBJECT(add_button), "clicked",
		GTK_SIGNAL_FUNC( add_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(del_button), "clicked",
		GTK_SIGNAL_FUNC( del_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(up_button), "clicked",
		GTK_SIGNAL_FUNC(up_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(down_button), "clicked",
		GTK_SIGNAL_FUNC(down_button_clicked_cb), selection);
	g_signal_connect(GTK_OBJECT(list_view), "key-press-event",
		GTK_SIGNAL_FUNC(list_view_key_press_event_cb), selection);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(scrolled_window), list_view);
	gtk_scrolled_window_set_shadow_type(
					GTK_SCROLLED_WINDOW(scrolled_window),
					GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show(scrolled_window);

	preset_box = gtk_vbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(preset_box), scrolled_window,
			TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(preset_box), button_box,
			FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(preset_box), 4);
	gtk_widget_show_all(preset_box);

	table = gtk_table_new(4, 2, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 4);
	gtk_widget_show(table);

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), TRUE);
	label_properties = gtk_label_new(_("Options"));
	label_presets =	gtk_label_new(_("Presets"));
	gtk_notebook_append_page
		(GTK_NOTEBOOK(notebook), preset_box, label_presets);
	gtk_box_pack_start
		(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);
	gtk_widget_show(notebook);

	GtkWidget *align = gtk_alignment_new(0, 0, 0.5, 0.5);
	gtk_widget_show(align);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 5, 5, 5);
	gtk_notebook_append_page
		(GTK_NOTEBOOK(notebook), align, label_properties);

	GtkWidget *properties = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(properties);
	gtk_container_add(GTK_CONTAINER(align), properties);

	// Device
	real_frame = gtk_frame_new(_("Device"));
	frame = gtk_alignment_new(0, 0, 0.5, 0.5);
	table = gtk_table_new(2, 2, FALSE);

	gtk_widget_show(frame);
	gtk_widget_show(table);
	gtk_widget_show(real_frame);

	gtk_alignment_set_padding(GTK_ALIGNMENT(frame), 0, 2, 2, 2);

	gtk_container_add(GTK_CONTAINER(properties), real_frame);
	gtk_container_add(GTK_CONTAINER(real_frame), frame);
	gtk_container_add(GTK_CONTAINER(frame), table);

	// V4L device
	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(_("V4L device"));
	device_entry = gtk_entry_new_with_max_length(MAX_DEVICE_NAME_LENGTH);
	gtk_entry_set_text(GTK_ENTRY(device_entry), data->device);

	gtk_widget_show(label);
	gtk_widget_show(device_entry);
	gtk_widget_show(hbox);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

	gtk_table_attach_defaults
		(GTK_TABLE(table), label, 0, 1, 0, 1);
	gtk_table_attach_defaults
		(GTK_TABLE(table), device_entry, 1, 2, 0, 1);

	// Auto-update the display
	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(_("Synchronize state with the card"));
	auto_update_display_yes = gtk_radio_button_new_with_label
		(NULL, _("yes"));

	gtk_widget_show(hbox);
	gtk_widget_show(label);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

	gtk_table_attach_defaults
		(GTK_TABLE(table), label, 0, 1, 1, 2);
	gtk_table_attach_defaults
		(GTK_TABLE(table), hbox, 1, 2, 1, 2);

	gtk_radio_button_set_group(GTK_RADIO_BUTTON(auto_update_display_yes),
		auto_update_display_group);
	auto_update_display_group = gtk_radio_button_get_group
		(GTK_RADIO_BUTTON(auto_update_display_yes));

	auto_update_display_no = gtk_radio_button_new_with_label
		(auto_update_display_group, _("no"));

	gtk_widget_show(auto_update_display_yes);
	gtk_widget_show(auto_update_display_no);

	gtk_box_pack_start
		(GTK_BOX(hbox), auto_update_display_no, FALSE, FALSE, 0);
	gtk_box_pack_start
		(GTK_BOX(hbox), auto_update_display_yes, FALSE, FALSE, 0);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
			(auto_update_display_yes), data->auto_update_display);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
			(auto_update_display_no), !data->auto_update_display);

	// Mouse-scrolling
	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(_("Mouse scrolling changes"));

	gtk_widget_show(hbox);
	gtk_widget_show(label);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 2, 3);

	frequency_button = gtk_radio_button_new_with_label(NULL,
		_("frequency"));
	gtk_radio_button_set_group(GTK_RADIO_BUTTON(frequency_button),
		scroll_group);
	scroll_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON
		(frequency_button));
	preset_button = gtk_radio_button_new_with_label(NULL, _("preset"));

	gtk_box_pack_start(GTK_BOX(hbox), frequency_button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), preset_button, FALSE, FALSE, 0);

	gtk_widget_show(frequency_button);
	gtk_widget_show(preset_button);

	gtk_radio_button_set_group
		(GTK_RADIO_BUTTON(preset_button), scroll_group);
	scroll_group =
		gtk_radio_button_get_group(GTK_RADIO_BUTTON(preset_button));

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(frequency_button),
		data->scroll == CHANGE_FREQ);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(preset_button),
		data->scroll == CHANGE_PRESET);

#ifdef ENABLE_ANTENNA_ICON
	// When off
	real_frame = gtk_frame_new(_("When radio is off"));
	frame = gtk_alignment_new(0, 0, 0.5, 0.5);
	GtkWidget *when_off = gtk_vbox_new(FALSE, 0);

	gtk_widget_show(when_off);
	gtk_widget_show(frame);
	gtk_widget_show(real_frame);

	gtk_alignment_set_padding(GTK_ALIGNMENT(frame), 0, 2, 2, 2);

	gtk_box_pack_start(GTK_BOX(properties), real_frame, FALSE, FALSE, 9);
	gtk_container_add(GTK_CONTAINER(real_frame), frame);
	gtk_container_add(GTK_CONTAINER(frame), when_off);

	label_when_off = gtk_radio_button_new_with_label
		(NULL, _("Show the label"));
	gtk_radio_button_set_group(GTK_RADIO_BUTTON(label_when_off),
							show_when_off);
	show_when_off = gtk_radio_button_get_group(GTK_RADIO_BUTTON
							(label_when_off));
	graphics_when_off = gtk_radio_button_new_with_label
		(show_when_off, _("Show graphics"));

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
			(label_when_off), data->show_label_when_off);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
			(graphics_when_off), !data->show_label_when_off);

	gtk_widget_show(label_when_off);
	gtk_widget_show(graphics_when_off);

	gtk_container_add(GTK_CONTAINER(when_off), label_when_off);
	gtk_container_add(GTK_CONTAINER(when_off), graphics_when_off);
#endif

	// When on
	real_frame = gtk_frame_new(_("When radio is running"));
	frame = gtk_alignment_new(0, 0, 0.5, 0.5);
	GtkWidget *when_on = gtk_vbox_new(FALSE, 0);
	GtkWidget *strength = gtk_hbox_new(FALSE, 0);

	gtk_widget_show(frame);
	gtk_widget_show(when_on);
	gtk_widget_show(strength);
	gtk_widget_show(real_frame);

	gtk_alignment_set_padding(GTK_ALIGNMENT(frame), 0, 2, 2, 2);

	gtk_box_pack_start(GTK_BOX(properties), real_frame, FALSE, FALSE, 9);
	gtk_container_add(GTK_CONTAINER(real_frame), frame);
	gtk_container_add(GTK_CONTAINER(frame), when_on);
	gtk_container_add(GTK_CONTAINER(when_on), strength);

	// Show the signal strength
	show_signal = gtk_check_button_new_with_label
		(_("Show signal strength"));
	gtk_widget_show(show_signal);
	gtk_container_add(GTK_CONTAINER(strength), show_signal);

#ifdef ENABLE_ANTENNA_ICON
	// Show the signal strength as
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);
	gtk_container_add(GTK_CONTAINER(strength), hbox);

	signal_type_bar = gtk_radio_button_new_with_label
		(signal_type_group, _("as a bar"));
	signal_type_group = gtk_radio_button_get_group
		(GTK_RADIO_BUTTON(signal_type_bar));
	signal_type_xpm = gtk_radio_button_new_with_label
		(signal_type_group, _("graphically"));

	gtk_widget_show(signal_type_bar);
	gtk_widget_show(signal_type_xpm);

	gtk_container_add(GTK_CONTAINER(strength), signal_type_bar);
	gtk_container_add(GTK_CONTAINER(strength), signal_type_xpm);
#endif

	// Show the label
	show_label = gtk_check_button_new_with_label
		(_("Show the label"));
	gtk_widget_show(show_label);
	gtk_container_add(GTK_CONTAINER(when_on), show_label);

	// Resolve presets
	GtkWidget *resolve_presets = gtk_check_button_new_with_label
					(_("Show preset names in label"));
	gtk_widget_show(resolve_presets);
	gtk_container_add(GTK_CONTAINER(when_on), resolve_presets);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(show_signal), data->show_signal);
#ifdef ENABLE_ANTENNA_ICON
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(signal_type_bar), !data->show_signal_graphical);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(signal_type_xpm), data->show_signal_graphical);
#endif
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(show_label), data->show_label);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
		(resolve_presets), data->resolve_presets_in_label);

	// Commands
	real_frame = gtk_frame_new(_("Commands"));
	frame = gtk_alignment_new(0, 0, 0.5, 0.5);
	table = gtk_table_new(2, 2, FALSE);

	gtk_widget_show(frame);
	gtk_widget_show(table);
	gtk_widget_show(real_frame);

	gtk_table_set_row_spacing(GTK_TABLE(table), 0, 2);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 2);
	gtk_alignment_set_padding(GTK_ALIGNMENT(frame), 0, 2, 2, 2);

	gtk_container_add(GTK_CONTAINER(frame), table);
	gtk_container_add(GTK_CONTAINER(real_frame), frame);
	gtk_box_pack_start(GTK_BOX(properties), real_frame, FALSE, FALSE, 0);

	// Post-startup command
	label = gtk_label_new(_("Run after startup"));
	startup_command_entry = gtk_entry_new_with_max_length
		(MAX_COMMAND_LENGTH);

	gtk_widget_show(label);
	gtk_widget_show(startup_command_entry);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

	gtk_entry_set_text
		(GTK_ENTRY(startup_command_entry), data->startup_command);

	gtk_table_attach_defaults
		(GTK_TABLE(table), label, 0, 1, 0, 1);
	gtk_table_attach_defaults
		(GTK_TABLE(table), startup_command_entry, 1, 2, 0, 1);

	// Post-shutdown command
	label = gtk_label_new(_("Run after shutdown"));
	shutdown_command_entry = gtk_entry_new_with_max_length
		(MAX_COMMAND_LENGTH);

	gtk_widget_show(label);
	gtk_widget_show(shutdown_command_entry);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

	gtk_entry_set_text
		(GTK_ENTRY(shutdown_command_entry), data->shutdown_command);

	gtk_table_attach_defaults
		(GTK_TABLE(table), label, 0, 1, 1, 2);
	gtk_table_attach_defaults
		(GTK_TABLE(table), shutdown_command_entry, 1, 2, 1, 2);

	g_signal_connect((gpointer) startup_command_entry, "changed",
			G_CALLBACK(radio_startup_command_changed), data);
	g_signal_connect((gpointer) shutdown_command_entry, "changed",
			G_CALLBACK(radio_shutdown_command_changed), data);
	g_signal_connect((gpointer) device_entry, "changed",
			G_CALLBACK(radio_device_changed), data);
	g_signal_connect(G_OBJECT(show_signal), "toggled",
			G_CALLBACK(radio_show_signal_changed), data);
	g_signal_connect(G_OBJECT(show_label), "toggled",
			G_CALLBACK(radio_show_label_changed), data);
#ifdef ENABLE_ANTENNA_ICON
	g_signal_connect(G_OBJECT(label_when_off), "toggled",
			G_CALLBACK(radio_show_label_when_off_changed), data);
	g_signal_connect(G_OBJECT(signal_type_bar), "toggled",
			G_CALLBACK(radio_signal_type_changed), data);
#endif
	g_signal_connect(G_OBJECT(resolve_presets), "toggled",
			G_CALLBACK(radio_resolve_presets_changed), data);
	g_signal_connect(G_OBJECT(auto_update_display_yes), "toggled",
			G_CALLBACK(radio_auto_update_display_changed), data);
	g_signal_connect(G_OBJECT(frequency_button), "toggled",
			G_CALLBACK(radio_scroll_type_changed), data);

	gtk_widget_show(dialog);
}

static void
radio_plugin_set_size(XfcePanelPlugin *plugin, int size, radio_gui *data)
{
	GtkWidget* bar = data->signal_bar;

	if (data->orientation == GTK_ORIENTATION_HORIZONTAL)
		gtk_widget_set_size_request(bar, BORDER, size - BORDER);
	else
		gtk_widget_set_size_request(bar, size - BORDER, BORDER);
}

static void
radio_set_orientation(XfcePanelPlugin *plugin, GtkOrientation orientation,
								radio_gui *gui)
{
	gui->orientation = orientation;
	gtk_container_remove(GTK_CONTAINER(gui->ebox), gui->outer_box);
	add_gui_elements(gui);

	update_label(gui);
}

static void
write_config_signal(XfcePanelPlugin *plugin, radio_gui *data)
{
	write_config(data, TRUE);
}

static void
write_config(radio_gui *data, gboolean save_presets)
{
	XfceRc *rc;
	int i;

	if (!(rc = get_config_rc()))
		return;

	xfce_rc_set_group	(rc, "radio plugin");

	xfce_rc_write_entry	(rc, "dev", data->device);
	xfce_rc_write_entry	(rc, "startup_cmd", data->startup_command);
	xfce_rc_write_entry	(rc, "shutdown_cmd", data->shutdown_command);

	xfce_rc_write_int_entry	(rc, "frq", data->freq);
	xfce_rc_write_int_entry	(rc, "scroll", data->scroll);
	xfce_rc_write_bool_entry(rc, "show_signal", data->show_signal);
	xfce_rc_write_bool_entry(rc, "update_display",
						data->auto_update_display);
	xfce_rc_write_bool_entry(rc, "show_label", data->show_label);
	xfce_rc_write_bool_entry(rc, "show_label_when_off",
						data->show_label_when_off);
	xfce_rc_write_bool_entry(rc, "resolve_presets_in_label",
						data->resolve_presets_in_label);
	xfce_rc_write_bool_entry(rc, "graphical_signal",
						data->show_signal_graphical);

	if (!save_presets)
	{
		xfce_rc_close(rc);
		return;
	}

	gchar *name;
	gint freq;
	GtkTreeIter iter;
	GtkTreeModel *presets = GTK_TREE_MODEL(data->presets);

	DBG("Writing configuration");
	gboolean valid = gtk_tree_model_get_iter_first(presets, &iter);

	// We start at 10 since there seems to be a problem loading keys which
	// are only 1 char long
	i = 10;
	while (valid)
	{
		if (i == 10)
		{
			xfce_rc_delete_group(rc, PRESET_NAMES, FALSE);
			xfce_rc_delete_group(rc, PRESET_FREQS, FALSE);
		}

		gtk_tree_model_get(presets, &iter, 0, &name, 1, &freq, -1);
		gchar *key = g_strdup_printf("%d", i);
		DBG("freq=%d, name=%s", freq, name);

		xfce_rc_set_group(rc, PRESET_NAMES);
		xfce_rc_write_entry(rc, key, name);
		xfce_rc_set_group(rc, PRESET_FREQS);
		xfce_rc_write_int_entry(rc, key, freq);

		g_free(name);
		g_free(key);
		i++;
		valid = gtk_tree_model_iter_next(presets, &iter);
	}
	xfce_rc_close(rc);
}

static void
import_old_config(XfcePanelPlugin *plugin, radio_gui *data)
{
	const char *value;
	char *file;
	XfceRc *rc;

	// Abort if there's nothing to be imported
	if (!(file = xfce_panel_plugin_lookup_rc_file(plugin)))
		return;

	DBG("Importing data from old conig file %s", file);

	rc = xfce_rc_simple_open(file, TRUE);
	g_free(file);

	if (!rc)
		return;

	xfce_rc_set_group(rc, "radio plugin");

	data->freq = xfce_rc_read_int_entry(rc, "frq", FREQ_INIT);
	data->scroll = xfce_rc_read_int_entry(rc, "scroll", CHANGE_FREQ);
	data->show_signal = xfce_rc_read_bool_entry(rc, "show_signal", TRUE);

	if ((value = xfce_rc_read_entry(rc, "dev", NULL)) && *value)
		strncpy(data->device, value, MAX_DEVICE_NAME_LENGTH);
	if ((value = xfce_rc_read_entry(rc, "startup_cmd", NULL)) && *value)
		strncpy(data->startup_command, value, MAX_COMMAND_LENGTH);
	if ((value = xfce_rc_read_entry(rc, "shutdown_cmd", NULL)) && *value)
		strncpy(data->shutdown_command, value, MAX_COMMAND_LENGTH);

	GtkTreeIter iter;
	GtkListStore *presets = data->presets;

	gchar** entries = xfce_rc_get_entries(rc, "presets");
	if (entries != NULL)
	{
		gchar **entry;
		entry = entries;

		xfce_rc_set_group(rc, "presets");

		while (*entry)
		{
			if ((value = xfce_rc_read_entry(rc, *entry, NULL))
								&& *value)
			{
				gint vfreq = atoi(*entry);
				const gchar *vname = value;

				DBG("freq=%d, name=%s", vfreq, vname);

				gtk_list_store_append(presets, &iter);
				gtk_list_store_set(presets, &iter, 0, vname,
								1, vfreq, -1);
			}
			entry++;
		}
	}
	g_strfreev(entries);
	xfce_rc_close(rc);

	write_config(data, TRUE);
}

static void
read_config(XfcePanelPlugin *plugin, radio_gui *data)
{
	const char *value;
	XfceRc *rc;

	if (!(rc = get_config_rc()))
		return;

	xfce_rc_set_group(rc, "radio plugin");

	data->freq = xfce_rc_read_int_entry
					(rc, "frq", FREQ_INIT);
	data->scroll = xfce_rc_read_int_entry
					(rc, "scroll", CHANGE_FREQ);
	data->show_signal = xfce_rc_read_bool_entry
					(rc, "show_signal", TRUE);
	data->auto_update_display = xfce_rc_read_bool_entry
					(rc, "update_display", TRUE);
	data->show_label = xfce_rc_read_bool_entry
					(rc, "show_label", TRUE);
	data->show_label_when_off = xfce_rc_read_bool_entry
					(rc, "show_label_when_off", TRUE);
	data->resolve_presets_in_label = xfce_rc_read_bool_entry
					(rc, "resolve_presets_in_label", TRUE);
	data->show_signal_graphical = xfce_rc_read_bool_entry
					(rc, "graphical_signal", FALSE);

	if ((value = xfce_rc_read_entry(rc, "dev", NULL)) && *value)
		strncpy(data->device, value, MAX_DEVICE_NAME_LENGTH);

	if ((value = xfce_rc_read_entry(rc, "startup_cmd", NULL)) && *value)
		strncpy(data->startup_command, value, MAX_COMMAND_LENGTH);

	if ((value = xfce_rc_read_entry(rc, "shutdown_cmd", NULL)) && *value)
		strncpy(data->shutdown_command, value, MAX_COMMAND_LENGTH);

	GtkTreeIter iter;
	g_return_if_fail(data->presets == NULL);
	data->presets = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
	GtkListStore *presets = data->presets;

	gchar** names = xfce_rc_get_entries(rc, PRESET_NAMES);
	gchar** freqs = xfce_rc_get_entries(rc, PRESET_FREQS);

	if (names == NULL || freqs == NULL) {
		import_old_config(plugin, data);
		xfce_rc_close(rc);
		return;
	}
	gchar **ename = names;
	gchar **efreq = freqs;

	while (*ename || *efreq)
	{
		xfce_rc_set_group(rc, PRESET_NAMES);
		const gchar *vname = xfce_rc_read_entry(rc, *ename,
							_(DEFAULT_NAME));
		xfce_rc_set_group(rc, PRESET_FREQS);
		gint vfreq = xfce_rc_read_int_entry(rc, *efreq, FREQ_MIN);
		DBG("freq=%d, name=%s", vfreq, vname);

		gtk_list_store_append(presets, &iter);
		gtk_list_store_set(presets, &iter, 0, vname, 1, vfreq, -1);
		ename++; efreq++;
	}
	g_strfreev(names);
	g_strfreev(freqs);
	xfce_rc_close(rc);
}

static void
radio_plugin_construct(XfcePanelPlugin *plugin)
{
	xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

	radio_gui *data = plugin_control_new(plugin);
	read_config(plugin, data);

	xfce_panel_plugin_menu_show_configure(plugin);

	g_signal_connect(plugin, "configure-plugin",
			G_CALLBACK(radio_plugin_create_options), data);
	g_signal_connect(plugin, "size-changed", G_CALLBACK
						(radio_plugin_set_size), data);
	g_signal_connect(plugin, "free-data", G_CALLBACK(radio_free), data);
	g_signal_connect(plugin, "save", G_CALLBACK(write_config_signal),
									data);
	g_signal_connect(plugin, "orientation-changed",
				G_CALLBACK(radio_set_orientation), data);
}

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL(radio_plugin_construct);
