/*#define DEBUG*/
/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>

#include <sys/types.h>

#include <sys/wait.h>

#include <limits.h>

#include <memory.h>

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#elif HAVE_VARARGS_H
#include <varargs.h>
#else
#error "no <stdarg.h> or <varargs.h>!"
#endif

#include <stdio.h>

#include <stdlib.h>

#include <string.h>

#include <time.h>

#include <unistd.h>

#include <regex.h>
#include <dirent.h>

#include <X11/Xlib.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>

#ifdef HAVE_LIBSM
#include <X11/SM/SMlib.h>
#endif
#include <libxfcegui4/libxfcegui4.h>


#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkprivate.h>

#include <dbh.h>

#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "basic_row.h"

#include "misc.h"
#include "modules.h"

#include "entry.h"
#include "icons.h"
#include "treeview.h"


extern root_t root[];

static gboolean diagnostics_disabled=FALSE;

    
static gchar *text_editors[]={
    "gvim",
    "emacs",
    "xedit",
    NULL
};
static gchar *text_types[]={
    "text/",
    "application/x-csh",
    "application/x-bsh",
    "application/x-ksh",
    "application/x-python",
    "application/x-shellscript",
    "application/x-tsh",
    "application/x-zsh",
    "application/xhtml+xml",
    "application/rtf",
    "application/x-cgi",
    "application/x-perl",
    "application/x-php",
    "application/x-theme",
    "application/x-trash",
    NULL
};

extern
SessionClient *session_client;

static 
void set_restart_command (int argc, char **argv){
#if 10
      int j,k;
      static gchar **saved_rstart=NULL;
      static gchar **rstart=NULL;

      if (!argv[1]) return;
      if (strcmp(argv[0],"xffm")!=0 && strcmp(argv[0],"xftree4")!=0) return;
      
      if (!saved_rstart){
	    int rstart_argc;
	    saved_rstart=session_client->restart_command;
	    for (rstart_argc=0;saved_rstart[rstart_argc];rstart_argc++);
	    rstart = (gchar **)malloc((argc+rstart_argc)*sizeof(gchar *));

	    for(j=0;j<argc;j++){
		rstart[j]=g_strdup(argv[j]);
	    }
	    for (k=1;j<argc+rstart_argc && session_client->restart_command[k];k++,j++){
		rstart[j]=g_strdup(session_client->restart_command[k]);
	    }
	    rstart[argc+rstart_argc-1]=NULL;
	    session_client->restart_command=rstart;
      } else {
	  g_free(rstart[1]);
	  rstart[1]=g_strdup(argv[1]);
      }

      return;
#endif
#if 0
#ifdef HAVE_LIBSM
      int i=0;
       SmProp prop1,*props[1];
       SmPropValue *vals = g_new (SmPropValue, argc);
       
       props[0] = &prop1;

	prop1.name = SmRestartCommand;
	prop1.type = SmLISTofARRAY8;
	prop1.vals = vals;
	for (i=0;i<argc;i++){
	    if (!argv[i]) {
		argc=i;
		break;
	    }
	    vals[i].value = argv[i];
	    vals[i].length = strlen(argv[i]);
	}
	prop1.num_vals = argc;
	
	printf("DBG:xxx 0x%x\n",session_client);
	printf("DBG:xxx 0x%x\n",session_client->session_connection);
	SmcSetProperties ((SmcConn) session_client->session_connection, 1, props);
	g_free(vals);
#endif
#endif

}


G_MODULE_EXPORT
XID get_xid (GtkTreeView * treeview)
{
    GtkWidget *w = WIDGET("vbox1");
    return GDK_WINDOW_XID(gtk_widget_get_parent_window(w));
}


G_MODULE_EXPORT
char *our_host_name (GtkTreeView * treeview)
{
    static char *name = NULL;
    unsigned char *property_data;
    unsigned long items, remaining;
    int actual_format;
    Atom atomo, actual_atom;

    if(name)
	return name;
    atomo = XInternAtom(GDK_DISPLAY(), "WM_CLIENT_MACHINE", FALSE);
    if(XGetWindowProperty(GDK_DISPLAY(), get_xid(treeview), atomo, 0, 255, FALSE, XA_STRING, &actual_atom, &actual_format, &items, &remaining, &property_data) == Success)
    {
	/*printf("dbg: property_data=%s\n",property_data); */
	name = g_strdup(property_data);
	XFree(property_data);
    }
    else
	name = g_strdup("localhost");

    return name;
}


/* I think this requieres static memory, hence _POSIX_PATH_MAX */
G_MODULE_EXPORT
void set_title (GtkTreeView * treeview, gchar **path)
{
    Atom atomo;
    char title[_POSIX_PATH_MAX], iconname[_POSIX_PATH_MAX], *hostname;
    char *word;
    static gboolean interned=FALSE;


    hostname = our_host_name(treeview);

    snprintf(title, _POSIX_PATH_MAX-1,"//%s%s", hostname, *path);
    title[_POSIX_PATH_MAX-1]=0;
    
    /* the following line is only for SMB paths, but shouldn't hurt for locals */

    word = strrchr(*path, '/');
    if(!word)
	snprintf(iconname, _POSIX_PATH_MAX-1,"%s: %s /", tree_details->argv[0],*path);
    else
	snprintf(iconname, _POSIX_PATH_MAX-1,"%s: %s /", tree_details->argv[0],word + 1);
    iconname[_POSIX_PATH_MAX-1]=0;
   
    gtk_window_set_title(
		    GTK_WINDOW(gtk_widget_get_toplevel(tree_details->window)), 
		    my_utf_string(title));
    gdk_window_set_icon_name(
		    gtk_widget_get_toplevel(tree_details->window)->window, 
		    my_utf_string(iconname));

      set_restart_command(2,tree_details->argv);
      if (interned) {
	static char *argv[3];
	argv[0]=tree_details->argv[0];
	argv[1]=tree_details->argv[1];
	argv[2]=NULL;
	
	XSetCommand(GDK_DISPLAY(), get_xid(treeview), argv, 2);
      }
      else 
      {
	interned=TRUE;
	atomo = XInternAtom(GDK_DISPLAY(), "WM_CLASS", FALSE);
	XChangeProperty(GDK_DISPLAY(), get_xid(treeview), atomo, 
		XA_STRING, 
		8, PropModeReplace, 
		(unsigned char *)"xffm", strlen("xffm") + 1);
	atomo = XInternAtom(GDK_DISPLAY(), "WM_COMMAND", FALSE);
	XChangeProperty(GDK_DISPLAY(), get_xid(treeview), atomo, XA_STRING, 
		8, PropModeReplace, 
		(unsigned char *)tree_details->argv[0], strlen(tree_details->argv[0]) + 1);

	if (tree_details->argv[1]) {
	    XChangeProperty(GDK_DISPLAY(), get_xid(treeview), 
		    atomo, XA_STRING,	/* GdkAtom type, */
		    8,		/* bit per data element: gint format */
	    	    PropModeAppend, 
		    (unsigned char *)tree_details->argv[1], strlen(tree_details->argv[1])+1);
    
	    XChangeProperty(GDK_DISPLAY(), get_xid(treeview), 
		    atomo, XA_STRING,	/* GdkAtom type, */
		    8,		/* bit per data element: gint format */
		    PropModeAppend, 
		    (unsigned char *)(*path), strlen(*path) + 1);
	}
    
      }
    

}


G_MODULE_EXPORT
gchar **text_type_OK (const gchar *path)
{ 
  const gchar *t=MIME_get_type((const gchar *)path, TRUE);
  if (t) {
      int i;
      for (i=0;text_types[i];i++) if (strncmp(text_types[i],t,strlen(text_types[i]))==0){
	  return text_editors;
      }
  }
  return NULL;
}


/*
 * This function converts a time value specified in seconds since 1970-01-01
 * to a string representation. It shows the date and the time if the point
 * if less then six month in the past and only the date otherwise.
 * The exact format must be provided by a translation string. 
 *
 * The function should be thread-save since there are not used static
 * (or even global) variables if the system provided a localtime_r() function.
 *
 * Arguments:     when:    time in seconds since 1970-01-01
 *                string:  the result char-array in which the string is placed
 *                length:  the length of the string-array
 *                
 * Return value:  string on success, NULL on failure
 */
G_MODULE_EXPORT
const 
gchar *
time_to_string (time_t when) 
{
      time_t          now             = time(NULL);
#ifdef HAVE_LOCALTIME_R
      struct tm       timestruct;
#endif
      struct tm*      timestruct_ptr;
      char	*formatstring;
      static gchar *s=NULL;
      gchar string[64];
      
#ifdef HAVE_MEMSET
      memset (string, 0, 64);
#else
      string[0]=0;
#endif
      
      formatstring = difftime(now, when) > 24*60*60*30*6
            /* strftime format for non-recent files (older than 6 months)  */
          ? _("%b %e  %Y")
            /* strftime format for recent files */
          : _("%b %e %H:%M");

#ifdef HAVE_LOCALTIME_R
      timestruct_ptr = &timestruct;
      localtime_r(&when, timestruct_ptr);
#else
      timestruct_ptr = localtime(&when);
#endif

      if (strftime(string, 64, formatstring, localtime(&when)) == 0) {
          return NULL;
      }
      g_free(s);
      s=g_strdup(my_utf_string(string));
      return (const gchar *)s;
}


G_MODULE_EXPORT
const gchar *
sizetag(off_t tama, gint count)
{
    gchar *tag = "B";
    static gchar *buf=NULL;
    long long utama=(long long)tama;

    g_free(buf);buf=NULL;
    if(utama >= 0)
    {
	if(utama >= (long long)1024 * 1024 * 1024)
	{
	    utama /= ((long long)1024 *1024 * 1024);
	    tag = "GB";
	}
	else if(utama >= 1024 * 1024)
	{
	    utama /= (1024 * 1024);
	    tag = "MB";
	}
	else if(utama >= 1024)
	{
	    utama /= 1024;
	    tag = "KB";
	}
	if(count <= 0){
	    /* format for size column of regular files */
	    buf=g_strdup_printf("%lld %s", utama, tag);
	}
	else if(count == 1){
	    /* format for size column of folders (one file) */
	    buf=g_strdup_printf(_("%d file, %lld %s."), count, utama, tag);
	}
	else {
	    /* format for size column of folders (more than one file) */
	    buf=g_strdup_printf(_("%d files, %lld %s."), count, utama, tag);
	}
    }
    else
    {
	if(count < 0){
	    buf=g_strdup_printf(" ");
	}
	else if(count == 1){
	    /* format for count of processed files (single file)*/
	    buf=g_strdup_printf(_("%d file"), count);
	} else {
	    /* format for count of processed files (multiple files)*/
	    buf=g_strdup_printf(_("%d files"), count);
	}
    }
    return (const gchar *)buf;
}


G_MODULE_EXPORT
const char *tod (void){
	time_t t=time(NULL);
	return ctime(&t);
}


G_MODULE_EXPORT
void count_selection (GtkTreeModel * treemodel, 
		GtkTreePath * treepath, 
		GtkTreeIter * iter, 
		gpointer data)
{
    tree_details->selectionOK++;
    return ;
}

G_MODULE_EXPORT
treestuff_t *get_treestuff (GtkTreeView *treeview){
    int i;
    for (i=0;i<TREECOUNT;i++){
	if (treeview == tree_details->treestuff[i].treeview) 
	    return (tree_details->treestuff)+i;
    }
    g_warning("treestuff != NULL");
    return NULL;
}

G_MODULE_EXPORT
gint get_tree_id (GtkTreeView *treeview){
    int i;
    for (i=0;i<TREECOUNT;i++){
      /*printf("DBG: treeviewcmp  0x%x==0x%x\n",tree_details->treestuff[i].treeview,treeview);*/
	if (treeview == tree_details->treestuff[i].treeview) 
	    return i;
    }
    /*g_error("treestuff[tree_id] != NULL");*/
    return -1;
}


static gboolean correct_selection;
static void query_selection(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    /*GtkTreeView *treeview = (GtkTreeView *) data;*/
    if(correct_selection) return;
    correct_selection=TRUE;
    return;
}

G_MODULE_EXPORT
GtkTreeView *get_bigger_treeview(void){
    GtkWidget *vpaned = WIDGET("hpaned1");
    if (gtk_paned_get_position((GtkPaned *)vpaned) <= vpaned->allocation.width * 0.50) {
	return tree_details->treestuff[0].treeview;
    } else return tree_details->treestuff[1].treeview;
}

G_MODULE_EXPORT
treestuff_t *get_only_visible_treestuff (void){
    GtkWidget *vpaned = WIDGET("hpaned1");
    if (gtk_paned_get_position((GtkPaned *)vpaned) <= vpaned->allocation.width * 0.10) {
	set_relative_tree_id(0);
	return tree_details->treestuff;
    } 
    if (gtk_paned_get_position((GtkPaned *)vpaned) >= vpaned->allocation.width *0.90) {
	set_relative_tree_id(1);
	return tree_details->treestuff+1;
    }
    return NULL;
}

G_MODULE_EXPORT
gboolean local_branch_is_visible (GtkTreeModel *treemodel){
    GtkTreeIter iter;
    tree_entry_t *en;
    if(gtk_tree_model_get_iter_first(treemodel, &iter)){
	do {  
	  gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
	  if (!en) continue;
	  if (IS_LOCAL_TYPE(en->type)) return TRUE;
	} while (gtk_tree_model_iter_next(treemodel, &iter));
    }
    return FALSE;	
}


static GtkTreeView *get_active_treeview(void){
    int i;
    treestuff_t *treestuff=get_only_visible_treestuff();
    GtkTreeView *treeview;
    GtkTreeSelection *selection;
    correct_selection=FALSE;
    
    
    if (treestuff) return (treestuff->treeview);
    for (i=0;i<TREECOUNT;i++) {
      treeview=tree_details->treestuff[i].treeview;
      selection = gtk_tree_view_get_selection(treeview);
      gtk_tree_selection_selected_foreach(selection, query_selection, (gpointer) treeview);
      if (correct_selection) return treeview;
    }
    return get_bigger_treeview();
}

G_MODULE_EXPORT
gint get_active_tree_id (void){
    gint i;
    GtkTreeView *treeview = get_active_treeview();
    i=get_tree_id(treeview);
    if (i < 0 || i >= TREECOUNT) 
	g_error("treeview != any(tree_details->treestuff[i].treeview)");
    return i;
}

/* new treeview selection algorithm */

static gint relative_one=0;

G_MODULE_EXPORT
gint get_ascending (void){
    return tree_details->treestuff[relative_one].ascending;
}

G_MODULE_EXPORT
gint get_relative_tree_id (void){
    return relative_one;
}
G_MODULE_EXPORT
void set_relative_tree_id (gint i){
    relative_one=i;
}

G_MODULE_EXPORT
gint get_relative_sort_column (void){
    return tree_details->treestuff[relative_one].sort_column;
}

G_MODULE_EXPORT
void on_treeview_column_click (GtkTreeViewColumn * column, gpointer data){
    set_relative_tree_id((gint)((long)data));
}

G_MODULE_EXPORT
gint set_relative_tree_id_from_model (GtkTreeModel *treemodel){
    int i;
    for (i=0;i<TREECOUNT;i++){
      /*printf("DBG: treeviewcmp  0x%x==0x%x\n",tree_details->treestuff[i].treeview,treeview);*/
	if (treemodel == tree_details->treestuff[i].treemodel) {
	    set_relative_tree_id(i);
	    return 1;
	}
    }
    g_warning("tree_id == NULL");
    return 1;
}

    

/* this function is to workaround a gtk bug in treeviews, where a 
 * while(gtk_events_pending()) will not break... 
 * Let's say 256 iterations suffice... 
 * But not always (if called from a timeout function),
 * so let's provide a blocking mechanism */
static gboolean pending_busy=FALSE;
G_MODULE_EXPORT
gboolean processing_pending (void){
	return pending_busy;
}

G_MODULE_EXPORT
void set_processing_pending (void){
	pending_busy=TRUE;
}
G_MODULE_EXPORT
void unset_processing_pending (void){
	pending_busy=FALSE;
}

G_MODULE_EXPORT
void process_pending_gtk (void){
  int i;
  static gboolean recursive=FALSE;
  /* breakout if recursive call */
  if (recursive) return;
  if (processing_pending()) return;
  recursive=TRUE;
  for (i=0;i<256;i++){
	  if (!gtk_events_pending()) break;
	  else gtk_main_iteration();
  }
  recursive=FALSE;  
}

G_MODULE_EXPORT
void disable_diagnostics (void){diagnostics_disabled=TRUE;};
G_MODULE_EXPORT
void enable_diagnostics (void){diagnostics_disabled=FALSE;};

G_MODULE_EXPORT
const gchar *get_xffm_home (void){
    static gchar *home=NULL;
    g_free(home); 
    home=NULL;    
    if (getenv("XFFM_HOME") && strlen( getenv("XFFM_HOME"))){
	    home=g_strdup(getenv("XFFM_HOME"));
    }
    if (!home) home=g_strdup(GETWD);
    if (!g_file_test(home,G_FILE_TEST_IS_DIR) || access(home,X_OK)!=0) {
	g_free(home);
	home=g_strdup("/");
    }
    return home;
}

static 
void purge_cache_files(const gchar *cd){
    DIR *directory;
    char *fullpath;
    struct dirent *d;
	gchar *xdg_dir;
	gchar *cache_dir;
	time_t now=time(NULL);
	xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	cache_dir = g_build_filename(xdg_dir,"xfce4","xffm",cd,NULL);  
	g_free(xdg_dir);
	if (!cache_dir) return;
	directory = opendir(cache_dir);
	if(!directory){
	    DBG("cannot open %s",cache_dir);
	    g_free(cache_dir);
	    return;
	}
	while((d = readdir(directory)) != NULL) 
	{
	    struct stat st;
	    if (strcmp(d->d_name,".")==0 || strcmp(d->d_name,"..")==0)continue;
	    fullpath = g_build_filename((gchar *)cache_dir, d->d_name,NULL);
	    if (strcmp(d->d_name,"..Wastebasket")==0){
		if (fork()==0){
		    execlp("rm","rm","-rf",fullpath,NULL);
		    _exit(123);
		}
	    }
	    if (stat(fullpath,&st) == 0){
		if (now - st.st_mtime > 3600*24*30) {
		    TRACE("removing %s",fullpath);
		    unlink(fullpath);
		}
		else TRACE("not removing %s: age is %d (%d days)",fullpath,(now - st.st_mtime),(now - st.st_mtime)/3600/24 );
	    }
	    g_free(fullpath);
        }
	closedir(directory);
        g_free(cache_dir);
	    
}
/* this could also delete tmp files in use by another xffm instance 
 * (very slight possibility)*/
G_MODULE_EXPORT
void cleanup_tmpfiles (void)
{
    DIR *directory;
    char *fullpath;
    struct dirent *d;
    const char *path=g_get_tmp_dir();
    directory = opendir(path);
    if(!directory) return;
    while((d = readdir(directory)) != NULL)
    {
	if(strncmp(d->d_name, "xffm",strlen("xffm"))==0 ||
	   strncmp(d->d_name, "tubopid",strlen("tubopid"))==0 ){
	   fullpath = g_build_filename((gchar *)path, d->d_name,NULL);
	   unlink(fullpath);
	   rmdir(fullpath);
	   g_free(fullpath);
	}
    }
    closedir(directory);
    purge_cache_files("smb");
    purge_cache_files("cache");
    purge_cache_files("thumbnails");
    return;
}

G_MODULE_EXPORT
void fork_function (void *data)
{
    char **argument = (char **)data;

    execvp(argument[0], argument);
    fprintf(stderr, "CHILD could not execvp: this should not happen");
    fprintf(stderr, "Do you have %s in your path?", argument[0]);
    fflush(NULL);
    usleep(500000);
    _exit(123);
}

G_MODULE_EXPORT
gint 
count_files (gchar *path)
{
    DIR *directory;
    struct dirent *dir;
    gint count = 0;

    directory = opendir(path);
    if(!directory)
	return -1;
    while((dir = readdir(directory)) != NULL)
    {
	if(strcmp(dir->d_name, ".") && strcmp(dir->d_name, ".."))		
		count++;
    }
    closedir(directory);
    return (count);
}

/* This function (count_hidden_files) is only used here... **/
static gint 
count_hidden_files(gchar *path)
{
    DIR *directory;
    struct dirent *dir;
    gint count = 0;

    directory = opendir(path);
    if(!directory) return -1;
    while((dir = readdir(directory)) != NULL)
    {
	if(strcmp(dir->d_name, ".") == 0 || 
	   strcmp(dir->d_name, "..") == 0 || 
	   strcmp(dir->d_name, "..Wastebasket") == 0)
	    continue;
	if(dir->d_name[0] == '.') count++;
    }
    closedir(directory);
    return (count);
}

G_MODULE_EXPORT
void set_entry_tag (GtkTreeView * treeview, tree_entry_t * en, off_t tama) 
{
    int hcount;

    hcount = count_hidden_files(en->path);
    if(hcount)
    {
	SET_HAS_HIDDEN(en->type);
    }
    if (en->tag) g_free(en->tag);
    if(SHOWS_HIDDEN(en->type))
	en->tag = g_strdup_printf("%s (%s %s)", FILENAME(en), sizetag(tama, en->count), _("Showing hidden."));
    else
    {
	if(hcount)
	    en->tag = g_strdup_printf("%s (%s %d %s)", FILENAME(en), sizetag(tama, en->count), hcount, _("hidden."));

	else
	    en->tag = g_strdup_printf("%s (%s %s)", FILENAME(en), sizetag(tama, en->count), _("No hidden."));
    }
}

 
G_MODULE_EXPORT
gboolean get_selectpath_iter(GtkTreeIter * iter, tree_entry_t ** en)
{
   gint tree_id = get_active_tree_id();
    GtkTreeModel *treemodel = tree_details->treestuff[tree_id].treemodel;
    GtkTreeSelection *selection = tree_details->treestuff[tree_id].selection;
    GtkTreeIter parent;
    tree_entry_t *c_en, *p_en = NULL;
    int caso = 0;

    if(!tree_details) return FALSE;
    if(!tree_details->window) return FALSE;


    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    if(!gtk_tree_selection_get_selected(selection, &treemodel, iter))
    {
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
	return FALSE;
    }
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &c_en, -1);
    if(gtk_tree_model_iter_parent(treemodel, &parent, iter))
    {
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, &p_en, -1);
    }

    if(IS_PATH(c_en->type) && IS_DIR(c_en->type))
	caso = 1;

    else if(IS_FIND_TYPE(c_en->type))
    {
	if(IS_DIR(c_en->type))
	    caso = 1;
	else if(p_en && IS_FILE(c_en->type) && IS_XF_FND(p_en->type))
	    caso = 0;
	else if(p_en && IS_FILE(c_en->type) && IS_DIR(p_en->type))
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_LOCAL_TYPE(c_en->type))
    {
	if(IS_DIR(c_en->type))
	    caso = 1;
	else if(p_en && (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)))
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_BOOKMARK_TYPE(c_en->type))
    {
	if(IS_DIR(c_en->type) || IS_ROOT_TYPE(c_en->type) ||
	   IS_NETDIR(c_en->subtype) || IS_XF_NETSHARE(c_en->subtype) ||
	   IS_XF_NETWS(c_en->subtype)  )
	    caso = 1;
	else if (p_en && 
		 (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)
		  || IS_NETFILE(c_en->subtype))
		) caso = 2;
	else
	    caso = 0;
    }
    else if(IS_TRASH_TYPE(c_en->type) || IS_RECENT_TYPE(c_en->type) 
	    || IS_FREQUENT_TYPE(c_en->type))
    {
	if(IS_ROOT_TYPE(c_en->type))
	    caso = 1;
	else if(p_en && (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)))
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_NETWORK_TYPE(c_en->type))
    {
	if(IS_ROOT_TYPE(c_en->type) || 
	   IS_XF_NETWG(c_en->subtype) ||
	   IS_XF_NETWS(c_en->subtype) ||
	   IS_NETDIR(c_en->subtype) ||
	   IS_XF_NETSHARE(c_en->subtype))
	    caso = 1;
	else if (p_en )
	    caso = 2;
	else
	    caso = 0;
    }
    else if (IS_FSTAB_TYPE(c_en->type))
    {
	if(IS_ROOT_TYPE(c_en->type)||IS_DIR(c_en->type))
	    caso = 1;
	else if(p_en && (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)))
	    		caso = 2;
	else caso = 0;
    }
	    
    /*printf("DBG: caso is %d\n",caso);*/
    if(caso)
    {
	if(caso == 2)
	{
	    *en = p_en;
	    *iter = parent;
	}
	else
	    *en = c_en;
    }
    else
	return FALSE;

    return TRUE;
}

G_MODULE_EXPORT
int get_the_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en, int which)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    if(!gtk_tree_model_get_iter_first(treemodel, iter)){
new_root:
#ifdef DEBUG
	   printf("DBG(misc.c):create_root_element(treeview,iter,which,g)\n");
#endif
	   create_root_element(treeview,iter,which,NULL);
   	   gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en, -1);
	   return TRUE;
    }
	   
	    
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en, -1);

    while((((*en)->type) & __TREE_MASK) != root[which].type)
    {
	if(!gtk_tree_model_iter_next(treemodel, iter))
	   goto new_root;
	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en, -1);
    }
    return TRUE;
}

G_MODULE_EXPORT
gboolean find_root(GtkTreeView * treeview, int which)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeIter iter;
    tree_entry_t *en;
    if(!gtk_tree_model_get_iter_first(treemodel, &iter)) return FALSE;
	   
	    
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);

    while(((en->type) & __TREE_MASK) != root[which].type)
    {
	if(!gtk_tree_model_iter_next(treemodel, &iter))
	   return FALSE;
	gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
    }
    return TRUE;
}

G_MODULE_EXPORT
int erase_the_root(GtkTreeView * treeview, int which)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeIter target;
    tree_entry_t *en;

    if(!gtk_tree_model_get_iter_first(treemodel, &target)){
	    return FALSE;
    } 
    gtk_tree_model_get(treemodel, &target, ENTRY_COLUMN, &en, -1);
    if (!en) return FALSE;

    while(((en->type) & __TREE_MASK) != root[which].type)
    {
	if(!gtk_tree_model_iter_next(treemodel, &target)){
	   return FALSE;
	}
	gtk_tree_model_get(treemodel, &target, ENTRY_COLUMN, &en, -1);
	if (!en) return FALSE;
    }
    remove_row(treemodel, &target, NULL, en);
    return TRUE;
}

static const gchar *abbreviate(char *string,int length){
    int i;
    static gchar *label = NULL;

    if (!string) return "";
    if(label) g_free(label);
    label = g_strdup(string);
    
    if(strlen(label) <= length) return my_utf_string((const gchar *)label);

    label[0] = '~';
    for(i = 1; i <= length; i++)
	label[i] = label[strlen(label) - (length - i)];
    return label;
}

G_MODULE_EXPORT
const gchar *abbreviate_end(gchar *string,int length){
    static gchar *label = NULL;

    if (!string) return "";
    if(label) g_free(label);
    label = g_strdup(string);
    
    if(strlen(label) <= length) return my_utf_string((const gchar *)label);

    label[length-1] = '~';
    label[length] = 0;
    return label;
}

G_MODULE_EXPORT
const char *abreviate(gchar *string)
{
     return abbreviate(string,35);
}


static gchar * my_cut_utf_string(gchar *s){
	static  gchar *u=NULL;
	gchar *v=g_strdup(my_utf_string(s));
	int max_len=48;
	if (getenv("XFFM_STATUS_LINE_LENGTH") && strlen( getenv("XFFM_STATUS_LINE_LENGTH"))) {
		if (is_number(getenv("XFFM_STATUS_LINE_LENGTH")))
		   max_len=(atoi(getenv("XFFM_STATUS_LINE_LENGTH")));
	}
	if (u) {
		g_free(u);
		u=NULL;
	}
        if (g_utf8_strlen(s,-1) > max_len){
	    u=g_utf8_offset_to_pointer (v,max_len-4);
	    *u=0;
	    u=g_strjoin ("",v,"...",NULL);
	    g_free(v);
	    v=NULL;
       } else u=v;
       return u;
} 

G_MODULE_EXPORT
gchar *randomTmpName(char *ext)
{
    gchar *fname;
    gchar *rname;
    rname = g_strconcat(g_get_tmp_dir(),G_DIR_SEPARATOR_S,"xffm.XXXXXX",NULL);
    D(printf("DBG: rname=%s\n",(rname)?rname:"null");)
    close(mkstemp(rname));
    if(ext == NULL) fname = g_strdup(rname);
    else fname = g_strconcat(rname,".",ext,NULL);
    D(printf("DBG: fname=%s\n",(fname)?fname:"null");)
    g_free(rname);
    return fname;
}

G_MODULE_EXPORT
const gchar *xffm_filename(gchar *path)
{
    gchar *name=NULL;
    
    g_free (name);

    if (!path) return "";
    name = g_path_get_basename(path);
    if (!name) return "";
    return (const gchar *)name;
}


static const gchar *plain_sizetag(int tama, int tamal)
{
    char *tag = "KB";
    char *tagl = "KB";
    static gchar *buf=NULL;

    g_free(buf);
    buf=NULL;
    if(tama >= 1024 * 1024)
    {
	tama /= (1024 *1024);
	tag = "GB";
    }
    else if(tama >= 1024)
    {
	tama /= 1024;
	tag = "MB";
    }

    if(tamal >= 1024 * 1024)
    {
	tamal /= (1024 *1024);
	tagl = "GB";
    }
    else if(tamal >= 1024)
    {
	tamal /= 1024;
	tagl = "MB";
    }

    buf=g_strdup_printf("%d %s / %d %s", tama, tag, tamal, tagl);

    return (const gchar *)buf;
}


G_MODULE_EXPORT
void set_progress_generic(int count, int total, int caso)
{
    static int smallcount = 1;

    gfloat fraction;

    GtkWidget *progress;

    if(!tree_details || !tree_details->window)
	return;
    progress = WIDGET("progressbar1");
    if(!progress)
	return;
    if(count == -1)
    {
	/*printf("DBG:updating pulse\n"); */
	if(total < 0 || (smallcount++ & (1 << 7)))
	{
	    gtk_progress_bar_pulse((GtkProgressBar *) progress);
    process_pending_gtk();
	    smallcount = 1;
	}
    }
    else
    {
	char texto[_POSIX_PATH_MAX];
	if(!total)
	    fraction = 0.0;
	else
	    fraction = (float)count / (float)total;
	if(fraction < 0 || fraction > 1)
	    return;
	gtk_progress_bar_set_fraction((GtkProgressBar *) progress, fraction);
	if(count >= total)
	    texto[0] = 0;
	else
	    switch (caso)
	    {
		case 1:
		    sprintf(texto, "%d / %d", count, total);
		    break;
		case 0:
		default:
		    sprintf(texto, "%s", plain_sizetag(count, total));
		    break;
	    }
	gtk_progress_bar_set_text((GtkProgressBar *) progress, texto);
    }
}


G_MODULE_EXPORT
void cursor_wait(void)
{
    static GdkCursor *cursor = NULL;
    if(!tree_details->window)
	return;
    if(!cursor)
	cursor = gdk_cursor_new(GDK_WATCH);
    gdk_window_set_cursor(tree_details->window->window, cursor);
    /*animation(treeview,TRUE); */
    gdk_flush();
}

G_MODULE_EXPORT
void cursor_reset(void)
{
    if(!tree_details->window)
	return;
    gdk_window_set_cursor(tree_details->window->window, NULL);
    /*animation(treeview,FALSE); */

}

G_MODULE_EXPORT
void set_font(GtkTreeModel *treemodel, GtkTreeIter * iterator)
{
    PangoFontDescription *desc;
    int size,ssize;
    tree_entry_t *en;
   
    if (!(tree_details->preferences & ENABLE_RESIZE_FONTS)) return;
       gtk_tree_model_get(treemodel, iterator, ENTRY_COLUMN, &en, -1);
    if (!en) return;   
    switch (tree_details->icon_size) {
	case 3: 
	    size = 16 * 1000;break;
	case 2:
	    size = 14 * 1000;break;
	case 1:
	    size = 12 * 1000;break;
	case 0:
	default:
	    size = 10 * 1000;break;
    }
    ssize = 10 * 1000;
    desc = pango_font_description_new();

#if 0
    if(IS_ROOT_TYPE(en->type))
	pango_font_description_set_family(desc, "times");
#endif

    pango_font_description_set_size(desc, size);

    /*printf("font size is=%d\n",pango_font_description_get_size(desc)); */
/*FIXME: all gtk_tree_store_set should reside in "basic" files*/
    gtk_tree_store_set((GtkTreeStore *) treemodel, iterator, FONT_COLUMN, desc, -1);
    pango_font_description_set_size(desc, ssize);
    gtk_tree_store_set((GtkTreeStore *) treemodel, iterator, SFONT_COLUMN, desc, -1);
    
    pango_font_description_free(desc);
}


G_MODULE_EXPORT
void hideit(GtkWidget * parent, const gchar * widget_name)
{
    GtkWidget *widget;
    if(!parent)
	return;
    if((widget = lookup_widget(parent,widget_name)) != NULL)
    {
	if(GTK_WIDGET_VISIBLE(widget))
	    gtk_widget_hide_all(widget);
    }

}
G_MODULE_EXPORT
void showit(GtkWidget * parent, const gchar * widget_name)
{
    GtkWidget *widget;
    if(!parent)
	return;
    if((widget = lookup_widget(parent,widget_name)) != NULL)
    {
	if(!GTK_WIDGET_VISIBLE(widget))
	    gtk_widget_show_all(widget);
    }

}

G_MODULE_EXPORT
void show_text(GtkWidget * parent)
{
    GtkWidget *vpaned;
    if(!parent)
	return;
    /*printf("DBG:show_text\n");*/
    vpaned = WIDGET("vpaned1");
    if (gtk_paned_get_position((GtkPaned *)vpaned) > vpaned->allocation.height * 0.80)
       gtk_paned_set_position(GTK_PANED(vpaned), vpaned->allocation.height * 0.80);
}

G_MODULE_EXPORT
void hide_text(GtkWidget * parent)
{
    GtkWidget *vpaned;
    vpaned = WIDGET("vpaned1");
    gtk_paned_set_position(GTK_PANED(vpaned), vpaned->allocation.height);
}

G_MODULE_EXPORT
const gchar *utf_2_local_string(const gchar *t){
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    const char *fc;
    gchar *from_codeset=NULL;

    
    g_free(s);
    s=NULL;
    g_get_charset(&fc);
    if (fc) from_codeset = g_strdup(fc);
    else from_codeset = g_strdup("ISO-8859-1");
    
    
    if (strcmp(from_codeset,"ASCII")==0){
	    g_free(from_codeset);
	    from_codeset = g_strdup("ISO-8859-1");
    }    
#ifdef DEBUG    
    printf("DBG: codeset for system is: %s\n",from_codeset);    
#endif
    
#ifdef DEBUG    
    printf("DBG: setting to local codeset-> %s\n",t); 
#endif
    if (strcmp(from_codeset,"UTF-8")==0) s = (gchar *)t;
    else {
	s = g_convert (t,strlen(t),from_codeset,"UTF-8",&r_bytes, &w_bytes, &error);
	if(error){
	    print_diagnostics("xfce/error",error->message,"\n",NULL);
	    g_error_free(error);
	}
    }
    g_free(from_codeset);
    return (const gchar *)s;
}

G_MODULE_EXPORT
const gchar *my_utf_string(const gchar *t)
{
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    unsigned char *c;
    const char *fc;
    gchar *from_codeset=NULL;
    
    if(!t) {
	g_warning("my_utf_string(NULL)");
	return "";
    }
    
    if (g_utf8_validate (t,-1,NULL)) {
#ifdef DEBUG    
    /*printf("DBG: valid utf8: %s\n",t);    */
#endif
	    return t;
    }
    
    /* so we got a non-UTF-8 */
    
    if (getenv("SMB_CODESET") && strlen( getenv("SMB_CODESET"))){
	    from_codeset=g_strdup(getenv("SMB_CODESET"));
    }
    else {
    	g_get_charset(&fc);
    	if (fc) from_codeset = g_strdup(fc);
    	else from_codeset = g_strdup("ISO-8859-1");
    }
    
/* hey, rsh will give you ANSI_X3.4-1968, so let's limit conversion with ISO */
/*    if (strcmp(from_codeset,"UTF-8")==0 || strcmp(from_codeset,"ASCII")==0){*/
    if (!strcmp(from_codeset,"ISO-")){
	    g_free(from_codeset);
	    from_codeset = g_strdup("ISO-8859-1");
    }    

#ifdef DEBUG    
    printf("DBG: codeset for system is: %s\n",from_codeset);    
#endif
    
    if(s) {
	    g_free(s);
	    s=NULL;
    }
#ifdef DEBUG    
    printf("DBG: utfing %s\n",t); 
#endif
    for(c = (unsigned char *)t; *c != 0; c++)
	if(*c < 32 && *c != '\n')
	    *c = ' ';
    s = g_convert (t,strlen(t),"UTF-8",from_codeset,&r_bytes, &w_bytes, &error);
    g_free(from_codeset);
    from_codeset=NULL;

    if(!s)
    {
	s=g_strdup(t);
	for(c = s; *c != 0; c++) if(*c > 128) *c = '?';
    }
    if(error){
        printf("DBG: %s. Codeset for system is: %s\n",
			error->message,from_codeset);
        printf("DBG: You should set the environment variable SMB_CODESET to ISO-8859-1\n");
	g_error_free(error);
    }
    return (const gchar *)s;
}


/*#define PANGO_IS_BUGGED*/

static void insert_string(GtkTextBuffer * buffer, const gchar *s)
{
    GtkTextIter start,end;
    if(s)
    {
#ifdef PANGO_IS_BUGGED
	gchar *o=NULL,*g=NULL;
	gtk_text_buffer_get_bounds(buffer,&start, &end);
	o=gtk_text_buffer_get_text(buffer,&start,&end,TRUE);
	g=g_strconcat(o,s,NULL);

	gtk_text_buffer_set_text  (buffer,(const gchar *)g,-1);
	g_free(o);
	g_free(g);
#else 
	gtk_text_buffer_get_bounds(buffer,&start, &end);
	gtk_text_buffer_insert(buffer, &end, s, -1);
#endif
    }
    return;
}

G_MODULE_EXPORT
gboolean is_number(char *p){
   char *c=p;
   if (!c || !strlen(c)) return FALSE;
   for (;*c;c++) {
	if (*c < '0' || *c > '9') return FALSE;
   }
   return TRUE;   
}


G_MODULE_EXPORT
void print_status_tmp(GdkPixbuf * icono, ...)
{
    va_list var_args;
    char *t;
    GtkTextIter start, end;
    GtkTextBuffer *buffer = gtk_text_view_get_buffer((GtkTextView *)WIDGET("status"));

    if(!tree_details->window)
	return;
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_delete(buffer, &start, &end);
    if(icono)
	gtk_text_buffer_insert_pixbuf(buffer, &end, icono);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_insert(buffer, &end, "  ", -1);
    va_start(var_args, icono);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_cut_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
  gdk_flush();
}



G_MODULE_EXPORT
void print_status(const gchar *id, ...)
{
    va_list var_args;
    char *t;
    GtkTextIter start, end;
    GtkTextBuffer *buffer = gtk_text_view_get_buffer((GtkTextView *)WIDGET("status"));

    if(!tree_details->window)
	return;
    gtk_text_buffer_set_text(buffer, " ", -1);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    if(id && icon_tell(0, id))
	gtk_text_buffer_insert_pixbuf(buffer, &end, icon_tell(0, id));
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_insert(buffer, &end, "  ", -1);
    va_start(var_args, id);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){	
	    insert_string(buffer, my_cut_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
  gdk_flush();
}


G_MODULE_EXPORT
void print_diagnostics(const gchar *id, ...)
{
    char *t;
    va_list var_args;
    GtkTextIter start, end;
    GtkTextMark *mark;
    GtkTextBuffer *buffer;

    if(diagnostics_disabled) return;
    if(!tree_details->window ){
	va_start(var_args, id);
	do
	{
	    t = va_arg(var_args, char *);
	    if(t && strlen(t)){
		printf(my_utf_string(t));
	    }
	}
	while(t);
	va_end(var_args);
	return;
    }
    buffer= gtk_text_view_get_buffer(GTK_TEXT_VIEW(WIDGET("diagnostics")));
    if (getenv("XFFM_DIAGNOSTICS_DISABLED") && strlen(getenv("XFFM_DIAGNOSTICS_DISABLED"))) return;
    if (!(tree_details->preferences & VERBOSE) && !id) return;
    if (id && strcmp(id,"nonverbose")==0) id=NULL;
    show_text((GtkWidget *)tree_details->window);
    /*printf("DBG:print_diagnostics\n");*/
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    if(id && icon_tell(0, id))
	gtk_text_buffer_insert_pixbuf(buffer, &end, icon_tell(0, id));

    va_start(var_args, id);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    mark = gtk_text_buffer_create_mark(buffer, "scrollmark", &end, FALSE);
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(WIDGET("diagnostics")), mark, 0.2,	/*gdouble within_margin, */
				 FALSE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(buffer, mark);
  gdk_flush();
}
G_MODULE_EXPORT
void print_diagnostics_tmp(GdkPixbuf * icono, ...)
{
    char *t;
    va_list var_args;
    GtkTextIter start, end;
    GtkTextMark *mark;
    GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(WIDGET("diagnostics")));

    if(!tree_details->window || diagnostics_disabled)
	return;
    /*printf("DBG:print_diagnostics_tmp\n");*/
    show_text((GtkWidget *)tree_details->window);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    if(icono)
	gtk_text_buffer_insert_pixbuf(buffer, &end, icono);
 
    va_start(var_args, icono);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);

    gtk_text_buffer_get_bounds(buffer, &start, &end);
    mark = gtk_text_buffer_create_mark(buffer, "scrollmark", &end, FALSE);
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(WIDGET("diagnostics")), mark, 0.2,	/*gdouble within_margin, */
				 FALSE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(buffer, mark);
  gdk_flush();
}


G_MODULE_EXPORT
void clear_diagnostics(void)
{
    GtkTextIter start, end;
    GtkTextBuffer *buffer;
    if(!tree_details->window)
	return;
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(WIDGET("diagnostics")));
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_delete(buffer, &start, &end);
    hide_text(tree_details->window);
}

G_MODULE_EXPORT
void show_stop(void)
{
    hideit(tree_details->window, "clear_text");
    showit(tree_details->window, "stop");
    /*printf("DBG: showing stop\n"); */
    process_pending_gtk();
}

G_MODULE_EXPORT
void hide_stop(void)
{
    hideit(tree_details->window, "stop");
    showit(tree_details->window, "clear_text");
    /*printf("DBG: hiding stop\n"); */
}

G_MODULE_EXPORT
GtkTreeIter *
get_iter_from_reference(	GtkTreeModel * treemodel, 
				GtkTreeRowReference *reference)
{
    static GtkTreeIter iter;
    GtkTreePath *treepath;
    if (!gtk_tree_row_reference_valid(reference)){
        g_warning("Invalid row reference");
	return NULL;
    } else if ((treepath = gtk_tree_row_reference_get_path(reference))==NULL){
	g_warning("gtk_tree_row_reference_get_path() == NULL");
	return NULL;
    }
    gtk_tree_model_get_iter(treemodel, &iter, treepath);
    gtk_tree_path_free(treepath);
    return &iter;
}

typedef struct dostext_t {
	unsigned char readable;
	unsigned char unreadable;
} dostext_t;

/* starts at 0xc0 */
static dostext_t dostext[]={
 {0xc0, 0xb7}, /*  */
 {0xc1, 0xb5}, /*  */
 {0xc2, 0xb6}, /*  */
 {0xc3, 0xc7}, /*  */
 {0xc4, 0x8e}, /*  */
 {0xc5, 0x8f}, /*  */
 {0xc6, 0x92}, /*  */
 {0xc7, 0x80}, /*  */
 {0xc8, 0xd4}, /*  */
 {0xc9, 0x90}, /*  */
 {0xca, 0xd2}, /*  */
 {0xcb, 0xd3}, /*  */
 {0xcc, 0xde}, /*  */
 {0xcd, 0xd6}, /*  */
 {0xce, 0xd7}, /*  */
 {0xcf, 0xd8}, /*  */
 {0xd0, 0xd1}, /*  */
 {0xd1, 0xa5}, /*  */
 {0xd2, 0xe3}, /*  */
 {0xd3, 0xe0}, /*  */
 {0xd4, 0xe2}, /*  */
 {0xd5, 0xe5}, /*  */
 {0xd6, 0x99}, /*  */
 {0xd7, 0x9e}, /*  */
 {0xd8, 0x9d}, /*  */
 {0xd9, 0xeb}, /*  */
 {0xda, 0xe9}, /*  */
 {0xdb, 0xea}, /*  */
 {0xdc, 0x9a}, /*  */
 {0xdd, 0xed}, /*  */
 {0xde, 0xe8}, /*  */
 {0xdf, 0xe1}, /*  */
 {0xe0, 0x85}, /*  */
 {0xe1, 0xa0}, /*  */
 {0xe2, 0x83}, /*  */
 {0xe3, 0xc6}, /*  */
 {0xe4, 0x84}, /*  */
 {0xe5, 0x86}, /*  */
 {0xe6, 0x91}, /*  */
 {0xe7, 0x87}, /*  */
 {0xe8, 0x8a}, /*  */
 {0xe9, 0x82}, /*  */
 {0xea, 0x88}, /*  */
 {0xeb, 0x89}, /*  */
 {0xec, 0x8d}, /*  */
 {0xed, 0xa1}, /*  */
 {0xee, 0x8c}, /*  */
 {0xef, 0x8b}, /*  */
 {0xf0, 0xd0}, /*  */
 {0xf1, 0xa4}, /*  */
 {0xf2, 0x95}, /*  */
 {0xf3, 0xa2}, /*  */
 {0xf4, 0x93}, /*  */
 {0xf5, 0xe4}, /*  */
 {0xf6, 0x94}, /*  */
 {0xf7, 0xf6}, /*  */
 {0xf8, 0x9b}, /*  */
 {0xf9, 0x97}, /*  */
 {0xfa, 0xa3}, /*  */
 {0xfb, 0x96}, /*  */
 {0xfc, 0x81}, /*  */
 {0xfd, 0xec}, /*  */
 {0xfe, 0xe7}, /*  */
 {0,0}
};


static void dos_txt (char *the_char,gboolean readable){
  unsigned char *c;
  dostext_t *t;
  for (c=(unsigned char *) the_char; c && c[0]!=0; c++)  {
	  t=dostext;
	  while (t->readable){
		  if (readable){
			if (c[0] == t->unreadable) {c[0] = t->readable; break;}
		  } else {
			if (c[0] == t->readable) {c[0] = t->unreadable; break;}
		  }
		  t++;	  
	  }
  }
}

G_MODULE_EXPORT
void ascii_readable (char *the_char) {
	dos_txt(the_char,TRUE);
}
G_MODULE_EXPORT
void ascii_unreadable (char *the_char) {
	dos_txt(the_char,FALSE);
}


