/*
 *  xfce4-volstatus-icon
 *
 *  Copyright (c) 2006 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <libxfcegui4/libxfcegui4.h>

#include <ghal/ghal.h>

#include "xfce-volstatus-icon.h"
#include "xfce-volstatus-dialog.h"
#include "xfce-volstatus-common.h"
#include "xfce-volstatus-marshal.h"

struct _XfceVolstatusIcon
{
    GtkStatusIcon parent;
    
    GHashTable *drives;
    GtkWidget *status_dialog;
};

typedef struct _XfceVolstatusIconClass
{
    GtkStatusIconClass parent;
    
    /*< signals >*/
    void (*drive_added)(XfceVolstatusIcon *icon,
                        GHalDrive *drive,
                        gboolean has_mounted_volume);
    void (*drive_removed)(XfceVolstatusIcon *icon,
                          GHalDrive *drive);
    void (*drive_status_changed)(XfceVolstatusIcon *icon,
                                 GHalDrive *drive,
                                 gboolean has_mounted_volume);
} XfceVolstatusIconClass;

typedef struct
{
    XfceVolstatusIcon *icon;
    GList *volumes;
} VolumeList;

enum
{
    SIG_DRIVE_ADDED = 0,
    SIG_DRIVE_REMOVED,
    SIG_DRIVE_STATUS_CHANGED,
    N_SIGS,
};

static void xfce_volstatus_icon_class_init(XfceVolstatusIconClass *klass);

static void xfce_volstatus_icon_init(XfceVolstatusIcon *icon);
static void xfce_volstatus_icon_finalize(GObject *obj);

static void xfce_volstatus_icon_activate(GtkStatusIcon *status_icon);
static void xfce_volstatus_icon_popup_menu(GtkStatusIcon *status_icon,
                                           guint button,
                                           guint activate_time);

static void xfce_volstatus_icon_menu_deactivate(GtkWidget *widget,
                                                gpointer user_data);
static void xfce_volstatus_icon_drive_activate(GtkMenuItem *mi,
                                               gpointer user_data);

static void xfce_volstatus_icon_find_mounted_drives(gpointer key,
                                                    gpointer value,
                                                    gpointer data);

static gboolean xfce_volstatus_icon_should_be_visible(XfceVolstatusIcon *icon);

static void xfce_volstatus_icon_volume_mounted(GHalVolume *volume,
                                               gpointer user_data);
static void xfce_volstatus_icon_volume_unmounted(GHalVolume *volume,
                                                 gpointer user_data);

static void xfce_volstatus_icon_disconnect_volume_signals(gpointer data,
                                                          gpointer user_data);

static void xfce_volstatus_icon_volume_list_destroy(VolumeList *vlist);

static guint xvsi_signals[N_SIGS] = { 0, };


G_DEFINE_TYPE(XfceVolstatusIcon, xfce_volstatus_icon, GTK_TYPE_STATUS_ICON)


static void
xfce_volstatus_icon_class_init(XfceVolstatusIconClass *klass)
{
    GObjectClass *object_class = (GObjectClass *)klass;
    GtkStatusIconClass *icon_class = (GtkStatusIconClass *)klass;
    
    object_class->finalize = xfce_volstatus_icon_finalize;
    
    icon_class->activate = xfce_volstatus_icon_activate;
    icon_class->popup_menu = xfce_volstatus_icon_popup_menu;
    
    xvsi_signals[SIG_DRIVE_ADDED] = g_signal_new("drive-added",
                                                 XFCE_TYPE_VOLSTATUS_ICON,
                                                 G_SIGNAL_RUN_LAST,
                                                 G_STRUCT_OFFSET(XfceVolstatusIconClass,
                                                                 drive_added),
                                                 NULL, NULL,
                                                 xfce_volstatus_marshal_VOID__OBJECT_BOOLEAN,
                                                 G_TYPE_NONE, 2,
                                                 GHAL_TYPE_DRIVE,
                                                 G_TYPE_BOOLEAN);
    
    xvsi_signals[SIG_DRIVE_REMOVED] = g_signal_new("drive-removed",
                                                   XFCE_TYPE_VOLSTATUS_ICON,
                                                   G_SIGNAL_RUN_LAST,
                                                   G_STRUCT_OFFSET(XfceVolstatusIconClass,
                                                                   drive_removed),
                                                   NULL, NULL,
                                                   g_cclosure_marshal_VOID__OBJECT,
                                                   G_TYPE_NONE, 1,
                                                   GHAL_TYPE_DRIVE);
    xvsi_signals[SIG_DRIVE_STATUS_CHANGED] = g_signal_new("drive-status-changed",
                                                          XFCE_TYPE_VOLSTATUS_ICON,
                                                          G_SIGNAL_RUN_LAST,
                                                          G_STRUCT_OFFSET(XfceVolstatusIconClass,
                                                                          drive_status_changed),
                                                          NULL, NULL,
                                                          xfce_volstatus_marshal_VOID__OBJECT_BOOLEAN,
                                                          G_TYPE_NONE, 2,
                                                          GHAL_TYPE_DRIVE,
                                                          G_TYPE_BOOLEAN);
}

static void
xfce_volstatus_icon_init(XfceVolstatusIcon *icon)
{
    icon->drives = g_hash_table_new_full(ghal_device_hash, ghal_device_equal,
                                         (GDestroyNotify)g_object_unref,
                                         (GDestroyNotify)xfce_volstatus_icon_volume_list_destroy);
}

static void
xfce_volstatus_icon_finalize(GObject *obj)
{
    XfceVolstatusIcon *icon = XFCE_VOLSTATUS_ICON(obj);
    
    if(icon->status_dialog) {
        g_object_remove_weak_pointer(G_OBJECT(icon->status_dialog),
                                     (gpointer)&icon->status_dialog);
        gtk_widget_destroy(icon->status_dialog);
    }
    
    g_hash_table_destroy(icon->drives);
    
    G_OBJECT_CLASS(xfce_volstatus_icon_parent_class)->finalize(obj);
}



static void
xfce_volstatus_icon_activate(GtkStatusIcon *status_icon)
{
    XfceVolstatusIcon *icon = XFCE_VOLSTATUS_ICON(status_icon);
    
    if(icon->status_dialog)
        gtk_window_present(GTK_WINDOW(icon->status_dialog));
    else {
        icon->status_dialog = xfce_volstatus_dialog_new(icon);
        gtk_widget_show(icon->status_dialog);
        g_signal_connect(G_OBJECT(icon->status_dialog), "response",
                         G_CALLBACK(gtk_widget_destroy), NULL);
        
        g_object_add_weak_pointer(G_OBJECT(icon->status_dialog),
                                  (gpointer)&icon->status_dialog);
    }
}

static void
xfce_volstatus_icon_popup_menu(GtkStatusIcon *status_icon,
                               guint button,
                               guint activate_time)
{
    XfceVolstatusIcon *icon = XFCE_VOLSTATUS_ICON(status_icon);
    GtkWidget *menu, *mi;
    GList *mounted_drives = NULL;
    
    menu = gtk_menu_new();
#if GTK_CHECK_VERSION(2, 12, 0)
    gtk_menu_set_screen(GTK_MENU(menu),
                        gtk_status_icon_get_screen(status_icon));
#endif
    gtk_widget_show(menu);
    g_signal_connect(G_OBJECT(menu), "deactivate",
                     G_CALLBACK(xfce_volstatus_icon_menu_deactivate), icon);
    
    g_hash_table_foreach(icon->drives, xfce_volstatus_icon_find_mounted_drives,
                         &mounted_drives);
    
    if(G_LIKELY(mounted_drives)) {
        GList *l;
        
        for(l = mounted_drives; l; l = l->next) {
            GHalDrive *drive = l->data;
            gchar *name, *label;
            
            name = ghal_drive_get_display_name(drive);
            label = g_strdup_printf(_("Safely remove \"%s\""), name);
            
            mi = gtk_menu_item_new_with_label(label);
            g_object_set_data(G_OBJECT(mi), "ghal-drive", drive);
            gtk_widget_show(mi);
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
            g_signal_connect(G_OBJECT(mi), "activate",
                             G_CALLBACK(xfce_volstatus_icon_drive_activate),
                             icon);
            
            g_free(name);
            g_free(label);
        }
        g_list_free(mounted_drives);
    } else {
        /* this shouldn't happen, but just to be safe... */
        mi = gtk_menu_item_new_with_label(_("(no removable drives)"));
        gtk_widget_set_sensitive(mi, FALSE);
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    }
    
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
                   gtk_status_icon_position_menu, icon,
                   button, activate_time);
}



static gboolean
xfce_volstatus_icon_menu_destroy_idled(gpointer data)
{
    gtk_widget_destroy(GTK_WIDGET(data));
    return FALSE;
}

static void
xfce_volstatus_icon_menu_deactivate(GtkWidget *widget,
                                    gpointer user_data)
{
    g_idle_add(xfce_volstatus_icon_menu_destroy_idled, widget);
}

static void
xfce_volstatus_icon_drive_activate(GtkMenuItem *mi,
                                   gpointer user_data)
{
    GHalDrive *drive = g_object_get_data(G_OBJECT(mi), "ghal-drive");
    xfce_volstatus_remove_drive(drive);
}

static void
xfce_volstatus_icon_drive_volume_added(GHalDrive *drive,
                                       GHalVolume *volume,
                                       gpointer user_data)
{
    XfceVolstatusIcon *icon = user_data;
    VolumeList *vlist = g_hash_table_lookup(icon->drives, drive);
    
    g_return_if_fail(vlist);
    g_return_if_fail(!g_list_find_custom(vlist->volumes, volume,
                                         ghal_device_compare));
    
    vlist->volumes = g_list_append(vlist->volumes,
                                   g_object_ref(G_OBJECT(volume)));
    
    g_signal_connect(G_OBJECT(volume), "mounted",
                     G_CALLBACK(xfce_volstatus_icon_volume_mounted), icon);
    g_signal_connect(G_OBJECT(volume), "unmounted",
                     G_CALLBACK(xfce_volstatus_icon_volume_unmounted), icon);
}

static void
xfce_volstatus_icon_drive_volume_removed(GHalDrive *drive,
                                         GHalVolume *volume,
                                         gpointer user_data)
{
    XfceVolstatusIcon *icon = user_data;
    VolumeList *vlist = g_hash_table_lookup(icon->drives, drive);
    GList *volume_node;
    
    g_return_if_fail(vlist);
    
    volume_node = g_list_find_custom(vlist->volumes, volume,
                                     ghal_device_compare);
    g_return_if_fail(volume_node);
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(volume_node->data),
                                         G_CALLBACK(xfce_volstatus_icon_volume_mounted),
                                         icon);
    g_signal_handlers_disconnect_by_func(G_OBJECT(volume_node->data),
                                         G_CALLBACK(xfce_volstatus_icon_volume_unmounted),
                                         icon);
    g_object_unref(G_OBJECT(volume_node->data));
    
    vlist->volumes = g_list_delete_link(vlist->volumes, volume_node);
}

static void
xfce_volstatus_icon_find_mounted_drives(gpointer key,
                                        gpointer value,
                                        gpointer data)
{
    GList **mounted_drives = data;
    GList *volumes = ((VolumeList *)value)->volumes, *l;
    
    for(l = volumes; l; l = l->next) {
        if(ghal_volume_is_mounted(GHAL_VOLUME(l->data))) {
            DBG("volume %s is mounted", ghal_device_peek_udi(GHAL_DEVICE(l->data)));
            *mounted_drives = g_list_prepend(*mounted_drives, key);
            break;
        }
    }
}

static gboolean
xfce_volstatus_icon_should_be_visible(XfceVolstatusIcon *icon)
{
    GList *mounted_drives = NULL;
    
    g_hash_table_foreach(icon->drives, xfce_volstatus_icon_find_mounted_drives,
                         &mounted_drives);
    if(mounted_drives) {
        g_list_free(mounted_drives);
        return TRUE;
    }
    
    return FALSE;
}

static void
xfce_volstatus_icon_ht_get_drives(gpointer key,
                                  gpointer value,
                                  gpointer user_data)
{
    GList **drives = user_data;
    *drives = g_list_prepend(*drives, key);
}

static void
xfce_volstatus_icon_volume_mounted(GHalVolume *volume,
                                   gpointer user_data)
{
    XfceVolstatusIcon *icon = user_data;
    GHalDevice *device;
    
    TRACE("entering");
    
    gtk_status_icon_set_visible(GTK_STATUS_ICON(icon), TRUE);
    
    device = ghal_volume_get_storage_device(volume);
    if(device) {
        if(GHAL_IS_DRIVE(device)) {
            gint already_has_mounted = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(device),
                                                                         "xvsi-already-has-mounted"));
            if(-1 == already_has_mounted) {
                g_signal_emit(G_OBJECT(icon),
                              xvsi_signals[SIG_DRIVE_STATUS_CHANGED], 0,
                              device, TRUE);
                g_object_set_data(G_OBJECT(device), "xvsi-already-has-mounted",
                                  GINT_TO_POINTER(1));
            }
        }
        g_object_unref(G_OBJECT(device));
    }
}

static void
xfce_volstatus_icon_volume_unmounted(GHalVolume *volume,
                                     gpointer user_data)
{
    XfceVolstatusIcon *icon = user_data;
    GHalDevice *device;
    
    TRACE("entering");
    
    if(!xfce_volstatus_icon_should_be_visible(icon))
        gtk_status_icon_set_visible(GTK_STATUS_ICON(icon), FALSE);
    
    device = ghal_volume_get_storage_device(volume);
    if(device) {
        if(GHAL_IS_DRIVE(device)) {
            gboolean has_mounted_volume = FALSE;
            GList *volumes = ghal_drive_list_volumes(GHAL_DRIVE(device)), *l;
            for(l = volumes; l; l = l->next) {
                if(ghal_volume_is_mounted(GHAL_VOLUME(l->data))) {
                    has_mounted_volume = TRUE;
                    break;
                }
            }
            g_list_foreach(volumes, (GFunc)g_object_unref, NULL);
            g_list_free(volumes);
            
            if(!has_mounted_volume) {
                g_signal_emit(G_OBJECT(icon),
                              xvsi_signals[SIG_DRIVE_STATUS_CHANGED], 0,
                              device, has_mounted_volume);
                g_object_set_data(G_OBJECT(device), "xvsi-already-has-mounted",
                                  GINT_TO_POINTER(-1));
            }
            
            g_object_unref(G_OBJECT(device));
        }
    }
}

static void
xfce_volstatus_icon_disconnect_volume_signals(gpointer data,
                                              gpointer user_data)
{
    g_signal_handlers_disconnect_by_func(G_OBJECT(data),
                                         G_CALLBACK(xfce_volstatus_icon_volume_mounted),
                                         user_data);
    g_signal_handlers_disconnect_by_func(G_OBJECT(data),
                                         G_CALLBACK(xfce_volstatus_icon_volume_unmounted),
                                         user_data);
}

static void
xfce_volstatus_icon_volume_list_destroy(VolumeList *vlist)
{
    g_list_foreach(vlist->volumes, (GFunc)xfce_volstatus_icon_disconnect_volume_signals,
                   vlist->icon);
    g_list_foreach(vlist->volumes, (GFunc)g_object_unref, NULL);
    g_list_free(vlist->volumes);
    g_free(vlist);
}



/*
 * public api
 */

GtkStatusIcon *
xfce_volstatus_icon_new(GdkScreen *screen)
{
    const gchar *icon_name = xfce_volstatus_get_icon_name();
    
    if(!screen)
        screen = gdk_display_get_default_screen(gdk_display_get_default());
    
    return g_object_new(XFCE_TYPE_VOLSTATUS_ICON,
#if GTK_CHECK_VERSION(2, 12, 0)
                        "screen", screen,
#endif
                        icon_name ? "icon-name" : "stock",
                        icon_name ? icon_name : GTK_STOCK_HARDDISK,
                        NULL);
}

void
xfce_volstatus_icon_add_drive(XfceVolstatusIcon *icon,
                              GHalDrive *drive)
{
    VolumeList *vlist;
    GList *volumes, *l;
    gboolean has_mounted_volume = FALSE;
    
    g_return_if_fail(XFCE_IS_VOLSTATUS_ICON(icon) && drive);
    
    if(g_hash_table_lookup(icon->drives, drive))
        return;
    
    volumes = ghal_drive_list_volumes(drive);
    for(l = volumes; l; l = l->next) {
        if(ghal_volume_is_mounted(GHAL_VOLUME(l->data)))
            has_mounted_volume = TRUE;
        
        g_signal_connect(G_OBJECT(l->data), "mounted",
                         G_CALLBACK(xfce_volstatus_icon_volume_mounted), icon);
        g_signal_connect(G_OBJECT(l->data), "unmounted",
                         G_CALLBACK(xfce_volstatus_icon_volume_unmounted),
                         icon);
    }
    
    vlist = g_new0(VolumeList, 1);
    vlist->icon = icon;
    vlist->volumes = volumes;
    g_hash_table_insert(icon->drives, g_object_ref(G_OBJECT(drive)), vlist);
    
    g_signal_connect(G_OBJECT(drive), "volume-added",
                     G_CALLBACK(xfce_volstatus_icon_drive_volume_added), icon);
    g_signal_connect(G_OBJECT(drive), "volume-removed",
                     G_CALLBACK(xfce_volstatus_icon_drive_volume_removed), icon);
    
    if(has_mounted_volume)
        gtk_status_icon_set_visible(GTK_STATUS_ICON(icon), TRUE);
    
    g_object_set_data(G_OBJECT(drive), "xvsi-already-has-mounted",
                      GINT_TO_POINTER(has_mounted_volume ? 1 : -1));
    g_signal_emit(G_OBJECT(icon), xvsi_signals[SIG_DRIVE_ADDED], 0, drive,
                  has_mounted_volume);
}

void
xfce_volstatus_icon_remove_drive(XfceVolstatusIcon *icon,
                                 GHalDrive *drive)
{
    g_return_if_fail(XFCE_IS_VOLSTATUS_ICON(icon) && drive);
    
    if(!g_hash_table_lookup(icon->drives, drive))
        return;
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(drive),
                                         G_CALLBACK(xfce_volstatus_icon_drive_volume_added),
                                         icon);
    g_signal_handlers_disconnect_by_func(G_OBJECT(drive),
                                         G_CALLBACK(xfce_volstatus_icon_drive_volume_removed),
                                         icon);
    
    /* g_hash_table_remove() will kill our only ref and might free the drive */
    g_object_ref(G_OBJECT(drive));
    g_hash_table_remove(icon->drives, drive);
    g_signal_emit(G_OBJECT(icon), xvsi_signals[SIG_DRIVE_REMOVED], 0, drive);
    g_object_unref(G_OBJECT(drive));
    
    if(!xfce_volstatus_icon_should_be_visible(icon))
        gtk_status_icon_set_visible(GTK_STATUS_ICON(icon), FALSE);
}

GList *
xfce_volstatus_icon_list_drives(XfceVolstatusIcon *icon)
{
    GList *drives = NULL;
    
    g_hash_table_foreach(icon->drives, xfce_volstatus_icon_ht_get_drives,
                         &drives);
    return g_list_reverse(drives);
}
