/******************************************************************************
 * Copyright 2004 Jasper Huijsmans <jasper@xfce.org>
 * 
 * Copyright (c) 2003 Tobias Henle <tobias@page23.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <gtk/gtk.h>

#include <panel/xfce.h>
#include <panel/plugins.h>

#include <libxfce4util/debug.h>
#include <libxfcegui4/dialogs.h>

typedef struct _XfceSystrayPlugin XfceSystrayPlugin;

struct _XfceSystrayPlugin
{
    GtkWidget *eventbox;
    GtkWidget *box;
    GtkWidget *traybox;
    GtkWidget *align[2];
    
    XfceSystemTray *systray;
    gboolean systray_reg;
};

static gboolean cb_register_systray (XfceSystrayPlugin * xsp);

static void cb_icon_docked (XfceSystemTray * tray,
                            GtkWidget * icon, XfceSystrayPlugin * xsp);

/*static void cb_unrealize (GtkContainer * container, XfceSystrayPlugin * xsp);*/
static void cb_icon_undocked (XfceSystemTray * tray,
                              GtkWidget * icon, XfceSystrayPlugin * xsp);
/*
static void cb_message_new (XfceSystemTray * tray,
                            GtkWidget * icon, guint timeout, guint buffer,
                            const gchar * message, XfceSystrayPlugin * xsp);
static void cb_message_clear (XfceSystemTray * tray,
                              GtkWidget * icon, guint none,
                              XfceSystrayPlugin * xsp);
*/

static void systray_attach_callback (Control * control,
                                     const char *signal, GCallback callback,
                                     gpointer data);
static gboolean systray_control_new (Control * control);
static void systray_free (Control * control);
static void systray_set_orientation (Control * control, int orientation);

/* CALLBACKS */

static void
cb_icon_docked (XfceSystemTray * tray, GtkWidget * icon,
                XfceSystrayPlugin * xsp)
{
    DBG (" ++ dock icon\n");
    
    if (xsp->systray_reg == TRUE)
    {
        gtk_widget_show (GTK_WIDGET (icon));
        gtk_box_pack_start (GTK_BOX (xsp->traybox), icon, FALSE, FALSE, 0);
    }
}

static void
cb_icon_undocked (XfceSystemTray * tray, GtkWidget * icon,
                  XfceSystrayPlugin * xsp)
{
    DBG (" ++ undock icon\n");
}

static gboolean
cb_register_systray (XfceSystrayPlugin * xsp)
{
    gboolean err;
    GError *error;

    TRACE ("register_systray");

    /* eureka we are registered                                           */
    if (xsp->systray_reg == TRUE)
        return FALSE;

    /* if there is another systray running wait for termination           */
    if (xfce_system_tray_check_running
        (DefaultScreenOfDisplay (GDK_DISPLAY ())) == TRUE)
        return TRUE;

    g_signal_connect (xsp->systray, "icon_docked",
                      G_CALLBACK (cb_icon_docked), xsp);
    g_signal_connect (xsp->systray, "icon_undocked",
                      G_CALLBACK (cb_icon_undocked), xsp);

    err = xfce_system_tray_register (xsp->systray,
                                     DefaultScreenOfDisplay (GDK_DISPLAY ()),
                                     &error);
    if (err == FALSE)
    {
        TRACE ("Unable to register system tray");
        xfce_err ("%s", error->message);
        /* if we are unable to register we should remember it to skip the   */
        /* unregister                                                       */
        xsp->systray_reg = FALSE;
        g_error_free (error);

        /* there was an error so we will disconnect                         */
        return FALSE;
    }
    else
    {
        xsp->systray_reg = TRUE;
        /* disconnect                                                       */
        return FALSE;
    }

    return TRUE;
}

static void
systray_attach_callback (Control * control,
                         const char *signal,
                         GCallback callback, gpointer data)
{
    /* attach the right btn click callback */
    XfceSystrayPlugin *xsp = (XfceSystrayPlugin *) control->data;

    TRACE ("signal:\'%s\' callback:%p data:%p", signal, callback, data);

    g_signal_connect (xsp->eventbox, signal, callback, data);
}

/* create panel control */

static gboolean
systray_control_new (Control * control)
{
    XfceSystrayPlugin *xsp;
    GtkWidget *sep[2];
    gboolean hor = settings.orientation == HORIZONTAL;

    xsp = g_new0 (XfceSystrayPlugin, 1);
    control->data = xsp;

    xsp->eventbox = gtk_event_box_new ();
    gtk_widget_show (xsp->eventbox);
    gtk_container_add (GTK_CONTAINER (control->base), xsp->eventbox);
    gtk_widget_set_size_request (control->base, -1, -1);
    
    if (hor)
        xsp->box = gtk_hbox_new (FALSE, 0);
    else
        xsp->box = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (xsp->box);
    gtk_container_add (GTK_CONTAINER (xsp->eventbox), xsp->box);
    
    xsp->align[0] = gtk_alignment_new (0.5, 0.5, 0.75, 0.75);
    gtk_container_set_border_width (GTK_CONTAINER (xsp->align[0]), 4);
    gtk_widget_show (xsp->align[0]);
    
    xsp->align[1] = gtk_alignment_new (0.5, 0.5, 0.75, 0.75);
    gtk_container_set_border_width (GTK_CONTAINER (xsp->align[1]), 4);
    gtk_widget_show (xsp->align[1]);

    if (hor)
    {
        xsp->traybox = gtk_hbox_new (FALSE, 0);
        sep[0] = gtk_vseparator_new ();
        sep[1] = gtk_vseparator_new ();
    }
    else
    {
        xsp->traybox = gtk_vbox_new (FALSE, 0);
        sep[0] = gtk_hseparator_new ();
        sep[1] = gtk_hseparator_new ();
    }
    gtk_widget_show (sep[0]);
    gtk_widget_show (sep[1]);
    gtk_widget_show (xsp->traybox);
    
    gtk_container_add (GTK_CONTAINER (xsp->align[0]), sep[0]);
    gtk_container_add (GTK_CONTAINER (xsp->align[1]), sep[1]);
    
    gtk_box_pack_start (GTK_BOX (xsp->box), xsp->align[0], FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (xsp->box), xsp->traybox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (xsp->box), xsp->align[1], FALSE, FALSE, 0);
    
    xsp->systray = xfce_system_tray_new ();
        
    if (cb_register_systray (xsp))
        return FALSE;

    return TRUE;
}

static void
systray_free (Control * control)
{
    XfceSystrayPlugin *xsp = (XfceSystrayPlugin *) control->data;

    TRACE ("systray_free");

    if (xsp->systray_reg == TRUE)
    {
        xfce_system_tray_unregister (xsp->systray);
        xsp->systray_reg = FALSE;
    }

    g_object_unref (xsp->systray);
    
    TRACE ("systray_free done");
}

static void
systray_set_orientation (Control * control, int orientation)
{
    XfceSystrayPlugin *xsp = (XfceSystrayPlugin *) control->data;
    gboolean hor = orientation == HORIZONTAL;
    GtkWidget *box, *traybox, *sep[2];

    TRACE ("systray_set_orientation");
    
    systray_free (control);

    if (hor)
    {
        box = gtk_hbox_new (FALSE, 0);
        traybox = gtk_hbox_new (FALSE, 5);
        sep[0] = gtk_vseparator_new ();
        sep[1] = gtk_vseparator_new ();
    }
    else
    {
        box = gtk_vbox_new (FALSE, 0);
        traybox = gtk_vbox_new (FALSE, 5);
        sep[0] = gtk_hseparator_new ();
        sep[1] = gtk_hseparator_new ();
    }
    
    gtk_widget_show (box);
    gtk_widget_show (traybox);
    gtk_widget_show (sep[0]);
    gtk_widget_show (sep[1]);
    
    gtk_widget_destroy (gtk_bin_get_child (GTK_BIN (xsp->align[0])));
    gtk_widget_destroy (gtk_bin_get_child (GTK_BIN (xsp->align[1])));

    gtk_widget_reparent (xsp->align[0], box);

    gtk_box_pack_start (GTK_BOX (box), traybox, FALSE, FALSE, 0);
    
    gtk_widget_reparent (xsp->align[1], box);

    gtk_container_add (GTK_CONTAINER (xsp->align[0]), sep[0]);
    gtk_container_add (GTK_CONTAINER (xsp->align[1]), sep[1]);
    
    gtk_widget_destroy (xsp->box);

    gtk_container_add (GTK_CONTAINER (xsp->eventbox), box);

    xsp->box = box;
    xsp->traybox = traybox;

    xsp->systray = xfce_system_tray_new ();
        
    cb_register_systray (xsp);
}

static void
systray_set_size (Control * control, int size)
{
}

G_MODULE_EXPORT void
xfce_control_class_init (ControlClass * cc)
{
    xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

    cc->name = "systray";
    cc->caption = _("Systemtray");

    cc->create_control = (CreateControlFunc) systray_control_new;

    cc->free = systray_free;
    cc->attach_callback = systray_attach_callback;
    cc->set_orientation = systray_set_orientation;
    cc->set_size = systray_set_size;

    control_class_set_unique (cc, TRUE);
}

XFCE_PLUGIN_CHECK_INIT
