/******************************************************************************
 * Copyright (c) 2003 Tobias Henle <tobias@page23.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <gtk/gtk.h>

#include <panel/global.h>
#include <panel/controls.h>
#include <panel/icons.h>
#include <panel/panel.h>
#include <panel/plugins.h>

#include <libxfce4util/debug.h>
#include <libxfcegui4/dialogs.h>

#include "xfce_hvbox.h"

typedef struct _XfceSystrayPlugin XfceSystrayPlugin;

struct _XfceSystrayPlugin
{
  Control		*control;
  GtkAlignment		*align;
  GtkBox		*systray_box;
  GtkWidget		*event_area;
  XfceSystemTray	*systray;
  GtkSeparator		*sep[2];
  GtkAlignment		*sep_box[2];
  gboolean		systray_reg;
  gboolean		base_rel;
  glong			signal;
  gint			timeout;
};

static XfceSystrayPlugin	*systray_plugin_new();
static gboolean			cb_register_systray(XfceSystrayPlugin *xsp);
//static void			cb_base_show(GtkWidget *widget, gpointer data);
static void			cb_icon_docked(XfceSystemTray *tray, 
    				  GtkWidget *icon, XfceSystrayPlugin *xsp);
static void			cb_unrealize(GtkContainer *container, 
				  XfceSystrayPlugin *xsp);
static void			cb_icon_undocked(XfceSystemTray *tray, 
    				  GtkWidget *icon, XfceSystrayPlugin *xsp);
static void			cb_message_new(XfceSystemTray *tray, 
    				  GtkWidget *icon, guint timeout, guint buffer, 
				  const gchar *message, XfceSystrayPlugin *xsp);
static void 			cb_message_clear(XfceSystemTray *tray, 
    				  GtkWidget *icon, guint none, 
				  XfceSystrayPlugin *xsp);
static void 			cb_selection_cleared(XfceSystemTray *tray, 
    				  XfceSystrayPlugin *xsp); 
/*static void		 	systray_create_options(Control *control, 
    				  GtkContainer *container, GtkWidget *revert, 
				  GtkWidget *done);
*/
static void 			systray_attach_callback(Control *control, 
    				  const char *signal, GCallback callback, 
				  gpointer data);
/*
static void 			systray_write_config(Control *control, 
    				  xmlNodePtr parent);
static void 			systray_read_config(Control *control, 
    				  xmlNodePtr node);
*/
static gboolean			systray_control_new(Control *control);
static void			systray_free(Control *control);
static void			systray_set_theme(Control *control, 
    				  const char *theme);
static void			systray_set_orientation(Control *control, 
    				  int orientation);

extern Settings settings;

static XfceSystrayPlugin* 
systray_plugin_new(Control *control)
{
  XfceSystrayPlugin	*xsp;
  GtkWidget *evbox;

  xsp			= g_malloc(sizeof(XfceSystrayPlugin));
  control->data		= xsp;
  TRACE("xsp:%p control->data:%p", xsp, control->data);
  xsp->control		= control;
  xsp->systray		= xfce_system_tray_new();
  xsp->timeout		= 0;
  xsp->align		= GTK_ALIGNMENT(gtk_alignment_new(0.5, 0.5, 1, 1));
  xsp->sep_box[0]	= GTK_ALIGNMENT(gtk_alignment_new(0.5, 0.5, 0.75, 0.75));
  xsp->sep_box[1]	= GTK_ALIGNMENT(gtk_alignment_new(0.5, 0.5, 0.75, 0.75));
  xsp->base_rel		= TRUE;
  xsp->sep[0]		= NULL;
  xsp->sep[1]		= NULL;
  xsp->systray_reg	= FALSE;

  if(settings.orientation == HORIZONTAL)
  {
    xsp->systray_box	= GTK_BOX(xfce_hvbox_new(FALSE, 7, 
			    XFCE_HVBOX_ORIENTATION_HORIZONTAL));
    xsp->event_area	= GTK_WIDGET(xfce_hvbox_new(FALSE, 2, 
			    XFCE_HVBOX_ORIENTATION_HORIZONTAL));
  }
  else
  {
    xsp->systray_box	= GTK_BOX(xfce_hvbox_new(FALSE, 7, 
			    XFCE_HVBOX_ORIENTATION_VERTICAL));
    xsp->event_area	= GTK_WIDGET(xfce_hvbox_new(FALSE, 2, 
			    XFCE_HVBOX_ORIENTATION_VERTICAL));
  }

  gtk_container_add(GTK_CONTAINER(control->base), GTK_WIDGET(xsp->align));
  gtk_container_add(GTK_CONTAINER(xsp->align), GTK_WIDGET(xsp->event_area));

  evbox = gtk_event_box_new();
  gtk_widget_show(evbox);
  gtk_container_set_border_width(GTK_CONTAINER(evbox), 3);
  gtk_box_pack_start(GTK_BOX(xsp->event_area), evbox, FALSE, FALSE, 0);
/*  gtk_box_pack_start(GTK_BOX(xsp->event_area), GTK_WIDGET(xsp->sep_box[0]), 
    TRUE, TRUE, 3);*/
  gtk_container_add(GTK_CONTAINER(evbox), GTK_WIDGET(xsp->sep_box[0]));
  
  gtk_box_pack_start(GTK_BOX(xsp->event_area), GTK_WIDGET(xsp->systray_box),
    TRUE, TRUE, 0);
  
  evbox = gtk_event_box_new();
  gtk_widget_show(evbox);
  gtk_container_set_border_width(GTK_CONTAINER(evbox), 3);
  gtk_box_pack_start(GTK_BOX(xsp->event_area), evbox, FALSE, FALSE, 0);
/*  gtk_box_pack_start(GTK_BOX(xsp->event_area), GTK_WIDGET(xsp->sep_box[1]),
    TRUE, TRUE, 0);*/
  gtk_container_add(GTK_CONTAINER(evbox), GTK_WIDGET(xsp->sep_box[1]));
  
  g_signal_connect(xsp->systray, "selection_cleared",
    G_CALLBACK(cb_selection_cleared), xsp);

  /* remove icons on unrealize, they will be readded */
  g_signal_connect(xsp->systray_box, "unrealize", G_CALLBACK(cb_unrealize), xsp);
  
  /* register returns true on no success					*/
  if(cb_register_systray(xsp)==TRUE)
    xsp->timeout = g_timeout_add(5 * 1000, (GSourceFunc)cb_register_systray, xsp);

  gtk_widget_show_all(GTK_WIDGET(control->base));

  systray_set_orientation(control, settings.orientation);
  
  TRACE("run");
  
  return xsp;
}

/********************************************************************************/
/* CALLBACKS									*/
/********************************************************************************/
/* GtkCallbacks									*/
/*
static void
cb_container_remove(GtkWidget *widget, gpointer data)
{
  gtk_container_remove(GTK_CONTAINER(data), widget);
}
*/

static void
cb_container_resize(GtkWidget *widget, gpointer data)
{
    GtkRequisition req;
    int s = icon_size[settings.size];
    
    gtk_widget_size_request(widget, &req);
    
    if (req.width > s || req.height > s)
    {
	gtk_widget_set_size_request(widget, s, s);
    }
}

static void
cb_unrealize(GtkContainer *container,  XfceSystrayPlugin *xsp)
{
  TRACE("systray unrealized");    

  gtk_container_foreach(container, (GtkCallback)gtk_widget_destroy, NULL);
}

/* XfceSystemTray Callbacks							*/
static void
cb_icon_docked(XfceSystemTray *tray, GtkWidget *icon, XfceSystrayPlugin *xsp)
{
#ifdef DEBUG_TRACE
  GtkRequisition 	request;
#endif
  
  TRACE("icon docked");
  
/*  gtk_widget_set_size_request(icon, icon_size[settings.size], 
    icon_size[settings.size]);*/

  gtk_box_pack_start(GTK_BOX(xsp->systray_box), GTK_WIDGET(icon), FALSE, FALSE, 0);

  gtk_widget_show(GTK_WIDGET(icon));

  gtk_widget_set_size_request (xsp->control->base, -1, -1);

#ifdef DEBUG_TRACE
  gtk_widget_size_request(GTK_WIDGET(xsp->event_area), &request);
  TRACE("real size width/height[%d/%d]", request.width, request.height);
#endif
}

static void
cb_icon_undocked(XfceSystemTray *tray, GtkWidget *icon, XfceSystrayPlugin *xsp)
{
  TRACE("icon undocked");
  
  gtk_container_remove(GTK_CONTAINER(xsp->systray_box), icon);
  
  gtk_widget_set_size_request (xsp->control->base, -1, -1);
}

static void
cb_message_new(
  XfceSystemTray *tray, 
  GtkWidget *icon, 
  guint timeout, 
  guint buffer,
  const gchar *message,
  XfceSystrayPlugin *xsp
) {
  TRACE("timeout:%d buffer:%d message:\'%s\'", timeout, buffer, message);
}

static void
cb_message_clear(
  XfceSystemTray *tray,
  GtkWidget *icon,
  guint none,
  XfceSystrayPlugin *xsp
) {
  TRACE("!");
}

static void
cb_selection_cleared(
  XfceSystemTray *tray,
  XfceSystrayPlugin *xsp
) {
  TRACE("!");
}

static gboolean
cb_register_systray (XfceSystrayPlugin *xsp)
{
  gboolean		err;
  GError		*error;
  
  TRACE("register_systray");

  /* heureka we are registered						*/
  if(xsp->systray_reg == TRUE)
    return FALSE;
 
  /* if there is another systray running wait for termination		*/
  if(xfce_system_tray_check_running(DefaultScreenOfDisplay(GDK_DISPLAY()))==TRUE)
    return TRUE;
    
  g_signal_connect(xsp->systray, "icon_docked",
    G_CALLBACK(cb_icon_docked), xsp);
  g_signal_connect(xsp->systray, "icon_undocked",
    G_CALLBACK(cb_icon_undocked), xsp);
  g_signal_connect(xsp->systray, "message_new",
    G_CALLBACK(cb_message_new), xsp);
  g_signal_connect(xsp->systray, "message_clear",
    G_CALLBACK(cb_message_clear), xsp);

  err 		= xfce_system_tray_register(xsp->systray, 
	            DefaultScreenOfDisplay(GDK_DISPLAY()), &error);
  if(err==FALSE)
  {
    TRACE("Unable to register system tray");
    xfce_err("Unable to register system tray: %s", error->message);
    /* if we are unable to register we should remember it to skip the 	*/
    /* unregister 							*/
    xsp->systray_reg	= FALSE;
    g_error_free(error);

    /* there was an error so we will disconnect				*/
    return FALSE;
  }
  else
  {
    xsp->systray_reg	= TRUE;
    /* disconnect							*/
    return FALSE;
  }

  return TRUE;
}

/* Contorl Callbacks								*/
/* the config options for the plugin						
static void
systray_create_options(
  Control      *control, 
  GtkContainer *container,
  GtkWidget    *revert,
  GtkWidget    *done
) {
  return;
}
*/
static void
systray_attach_callback(
  Control    *control,
  const char *signal,
  GCallback  callback, 
  gpointer   data
) {
  /* attach the right btn click callback					*/
  XfceSystrayPlugin *xsp 	= (XfceSystrayPlugin *)control->data;

  TRACE("signal:\'%s\' callback:%p data:%p", signal, callback, data);

  /* only the two separator boxes. sorry the right btn click is used by		*/
  /* the systray icons :(							*/
  g_signal_connect(GTK_WIDGET(xsp->sep_box[0])->parent, signal, callback, data);
  g_signal_connect(GTK_WIDGET(xsp->sep_box[1])->parent, signal, callback, data);

  return;
}

#if 0
static void
systray_write_config(Control * control, xmlNodePtr parent)
{
  /* do nothing									*/
  return;
}

static void
systray_read_config(Control *control, xmlNodePtr node)
{
  /* do nothing									*/
  return;
}
#endif

/* create pannel control							*/

/* ensure uniqueness (there can be only one ...) */
static gboolean systray_control_created = FALSE;

static gboolean
systray_control_new(Control *control)
{
  XfceSystrayPlugin *xsp;

  if (systray_control_created)
      return FALSE;
  
  xsp				= systray_plugin_new(control);
  control->data			= xsp;
  TRACE("xsp:%p control->data:%p", xsp, control->data);
  control->with_popup		= FALSE;
  gtk_widget_set_size_request (control->base, -1, -1);

  systray_control_created = TRUE;
  return TRUE;
}
  
static void
systray_free(Control *control)
{
  XfceSystrayPlugin *xsp	= (XfceSystrayPlugin *)control->data;

  TRACE("systray_free");

//  g_return_if_fail(xsp->base_rel == TRUE);
  
  if(xsp->timeout!=0)
    g_source_remove(xsp->timeout);

  if(xsp->systray_reg == TRUE)
  {
    xfce_system_tray_unregister(xsp->systray);
    g_object_unref(xsp->systray);
  }

  systray_control_created = FALSE;
  TRACE("systray_free done");

  return;
}

static void
systray_set_theme(Control *control, const char *theme)
{
  return;
}

static void
systray_set_orientation(Control *control, int orientation)
{
  XfceSystrayPlugin	*xsp	= (XfceSystrayPlugin *)control->data;

  TRACE("systray_set_orientation");

  g_return_if_fail(xsp->base_rel == TRUE);
  
  if(xsp == NULL)
    return;

  if(gtk_bin_get_child(GTK_BIN(xsp->sep_box[0]))!=NULL)
    gtk_container_remove(GTK_CONTAINER(xsp->sep_box[0]), 
      gtk_bin_get_child(GTK_BIN(xsp->sep_box[0])));
  
  if(gtk_bin_get_child(GTK_BIN(xsp->sep_box[1]))!=NULL)
    gtk_container_remove(GTK_CONTAINER(xsp->sep_box[1]), 
      gtk_bin_get_child(GTK_BIN(xsp->sep_box[1])));

  if(orientation == HORIZONTAL)
  {
    xsp->sep[0] = GTK_SEPARATOR(gtk_vseparator_new());
    xsp->sep[1] = GTK_SEPARATOR(gtk_vseparator_new());
    xfce_hvbox_set_orientation(XFCE_HVBOX(xsp->systray_box), 
      XFCE_HVBOX_ORIENTATION_HORIZONTAL); 
    xfce_hvbox_set_orientation(XFCE_HVBOX(xsp->event_area), 
      XFCE_HVBOX_ORIENTATION_HORIZONTAL); 
  }
  else
  {
    xfce_hvbox_set_orientation(XFCE_HVBOX(xsp->systray_box), 
      XFCE_HVBOX_ORIENTATION_VERTICAL); 
    xfce_hvbox_set_orientation(XFCE_HVBOX(xsp->event_area), 
      XFCE_HVBOX_ORIENTATION_VERTICAL); 
    xsp->sep[0] = GTK_SEPARATOR(gtk_hseparator_new());
    xsp->sep[1] = GTK_SEPARATOR(gtk_hseparator_new());
  }

  gtk_container_add(GTK_CONTAINER(xsp->sep_box[0]), GTK_WIDGET(xsp->sep[0]));
  gtk_container_add(GTK_CONTAINER(xsp->sep_box[1]), GTK_WIDGET(xsp->sep[1]));

  gtk_widget_show_all(GTK_WIDGET(xsp->sep_box[0]));
  gtk_widget_show_all(GTK_WIDGET(xsp->sep_box[1]));
  
  gtk_widget_set_size_request (control->base, -1, -1);
  
  return;
};

static void
systray_set_size(Control *control, int size)
{
  XfceSystrayPlugin	*xsp	= (XfceSystrayPlugin *)control->data;

  g_return_if_fail(xsp->base_rel == TRUE);
  
  gtk_container_forall(GTK_CONTAINER(xsp->systray_box), cb_container_resize, 
    (gpointer)NULL);

  return;
}
  
G_MODULE_EXPORT void
xfce_control_class_init(ControlClass *cc)
{
#ifdef ENABLE_NLS
    /* This is required for UTF-8 at least - Please don't remove it */
    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
    textdomain (GETTEXT_PACKAGE);
#endif

  cc->name		= "systray";
  cc->caption		= "Systemtray";

  cc->create_control	= (CreateControlFunc)systray_control_new;

  cc->free		= systray_free;
/*  cc->read_config	= systray_read_config;*/
  cc->read_config	= NULL;
/*  cc->write_config	= systray_write_config;*/
  cc->write_config	= NULL;
  cc->attach_callback	= systray_attach_callback;

/*  cc->create_options	= (gpointer)systray_create_options;*/

  cc->set_theme		= systray_set_theme;
  cc->set_orientation	= systray_set_orientation;
  cc->set_size		= systray_set_size;
}

XFCE_PLUGIN_CHECK_INIT
