#ifndef XFC_GTK_PRINT_OPERATION_HH
#define XFC_GTK_PRINT_OPERATION_HH 1

/**
   GtkPrintOperation interface
*/

#ifndef XFC_GTK_WINDOW_HH
#include <xfc/gtk/window.hh>
#endif

#ifndef XFC_GTK_PRINT_PRINTCONTEXT_HH
#include <xfc/gtk/printcontext.hh>
#endif

#ifndef XFC_GTK_PRINT_SETTING_HH
#include <xfc/gtk/printsetting.hh>
#endif

#ifndef XFC_GTK_PRINT_OPERATION_PREVIEW_HH
#include "xfc/gtk/printoperationpreview.hh"
#endif

namespace Xfc {
    namespace Gtk {
        class PrintSettings;

        enum PrintOperationResult {
            ERROR       = GTK_PRINT_OPERATION_RESULT_ERROR,
            APPLY       = GTK_PRINT_OPERATION_RESULT_APPLY,
            CANCEL      = GTK_PRINT_OPERATION_RESULT_CANCEL,
            IN_PROGRESS = GTK_PRINT_OPERATION_RESULT_IN_PROGRESS
        };
        
        enum PrintOperationAction {
            ACTION_PRINT_DIALOG = GTK_PRINT_OPERATION_ACTION_PRINT_DIALOG,
            ACTION_PRINT        = GTK_PRINT_OPERATION_ACTION_PRINT,
            ACTION_PREVIEW      = GTK_PRINT_OPERATION_ACTION_PREVIEW,
            ACTION_EXPORT       = GTK_PRINT_OPERATION_ACTION_EXPORT
        };

        enum PrintStatus {
            INITIAL          = GTK_PRINT_STATUS_INITIAL,
            PREPARING        = GTK_PRINT_STATUS_PREPARING,
            GENERATING_DATA  = GTK_PRINT_STATUS_GENERATING_DATA,
            SENDING_DATA     = GTK_PRINT_STATUS_SENDING_DATA,
            PENDING          = GTK_PRINT_STATUS_PENDING,
            PENDING_ISSUE    = GTK_PRINT_STATUS_PENDING_ISSUE,
            PRINTING         = GTK_PRINT_STATUS_PRINTING,
            FINISHED         = GTK_PRINT_STATUS_FINISHED,
            FINISHED_ABORTED = GTK_PRINT_STATUS_FINISHED_ABORTED
        };

        class PrintOperation : public PrintOperationPreview {
            friend class G::Object;

            PrintOperation(const PrintOperation &);
           
            PrintOperation &operator=(const PrintOperation &);
        protected:
            /// @name Constructors
            /// @{
                
            PrintOperation();
            
            explicit PrintOperation(GtkPrintOperation *op, bool owns_reference = false);
            
            virtual ~PrintOperation();
            
            /// @}
            /// @name Signal Prototypes
            /// @{
            
            typedef G::Signal<void,PrintContext&> BeginPrintSignalType;
            typedef G::SignalProxy<G::TypeInterface, BeginPrintSignalType> BeginPrintSignalProxy;
            static const BeginPrintSignalType begin_print_signal;
            ///< Begin print signal proxy

            typedef G::Signal<void,Widget*> CreateCustomWidgetSignalType;
            typedef G::SignalProxy<G::TypeInterface, CreateCustomWidgetSignalType> CreateCustomWidgetSignalProxy;
            static const CreateCustomWidgetSignalType create_custom_widget_signal;
            ///< Create custom widget signal

            typedef G::Signal<void,Widget*> CustomWidgetApplySignalType; 
            typedef G::SignalProxy<G::TypeInterface, CustomWidgetApplySignalType> CustomWidgetApplySignalProxy;
            static const CustomWidgetApplySignalType custom_widget_apply_signal;
            ///< 

            typedef G::Signal<void,PrintOperationResult&> DoneSignalType;
            typedef G::SignalProxy<G::TypeInterface,DoneSignalType> DoneSignalProxy;
            static const DoneSignalType done_signal;

            typedef G::Signal<void,PrintContext&,int> DrawPageSignalType;
            typedef G::SignalProxy<G::TypeInterface,DrawPageSignalType> DrawPageSignalProxy;
            static const DrawPageSignalType draw_page_signal;

            typedef G::Signal<void,PrintContext&> EndPrintSignalType;
            typedef G::SignalProxy<G::TypeInterface,EndPrintSignalType> EndPrintSignalProxy;
            static const EndPrintSignalType end_print_signal;
            
            typedef G::Signal<bool,PrintContext&> PaginateSignalType;
            typedef G::SignalProxy<G::TypeInterface,PaginateSignalType> PaginateSignalProxy;
            static const PaginateSignalType paginate_signal;

            typedef G::Signal<bool,PrintOperationPreview&,PrintContext&,Window&> PreviewSignalType;
            typedef G::SignalProxy<G::TypeInterface,PreviewSignalType> PreviewSignalProxy;
            static const PreviewSignalType preview_signal;

            typedef G::Signal<void,PrintContext&,int,PageSetup&> RequestPageSetupSignalType;
            typedef G::SignalProxy<G::TypeInterface,RequestPageSetupSignalType> RequestPageSetupSignalProxy;
            static const RequestPageSetupSignalType request_page_setup;

            typedef G::Signal<void> StatusChangedSignalType;
            typedef G::SignalProxy<G::TypeInterface,StatusChangedSignalType> StatusChangedSignalProxy;
            static const StatusChangedSignalType status_changed_signal;
            
        public:
            /// @}
            /// @name Accessors
            /// @{
            
            operator GtkPrintOperation* () const;

            GtkPrintOperation* gtk_print_operation() const;
            ///< Get a pointer to the GtkPrintOperation structure.
            
            PrintStatus get_status();
            String get_status_string();
            bool is_finished();
            
            /// @}
            /// @name Methods
            /// @{

            void set_allow_async(bool allow_async);
            void set_default_page_setup( PageSetup *default_page_setup);
            PageSetup* get_default_page_setup();
            void set_print_settings( PrintSettings *print_settings );
            PrintSettings* get_print_settings();
            void set_job_name( const String &job_name );
            void set_n_pages( int n_pages );
            void set_current_page( int current_page);
            void set_use_full_page( bool full_page);
            void set_unit(Gtk::Unit unit);
            void set_export_filename( const String &filename);
            void set_show_progress( bool show_progress);
            void set_track_print_status( bool track_status);
            void set_custom_tab_label( const String &label);
            PrintOperationResult run( PrintOperationAction action, Window *parent = NULL);
            void cancel();

            /// @}
            /// @name Signal Proxies
            /// @{

        	const BeginPrintSignalProxy signal_begin_print();
            ///< Emitted after the user has finished changing print settings in the dialog, 
            ///< before the actual rendering starts.
            ///< A typical use for ::begin-print is to use the parameters 
            ///< from the GtkPrintContext and paginate the document accordingly, and then 
            ///< set the number of pages with gtk_print_operation_set_n_pages().

            const CreateCustomWidgetSignalProxy signal_create_custom_widget();
            ///< Emitted when displaying the print dialog. If you return a widget in a handler 
            ///< for this signal it will be added to a custom tab in the print dialog. You 
            ///< typically return a container widget with multiple widgets in it.
            ///< The print dialog owns the returned widget, and its lifetime isn't controlled 
            ///< by the app. However, the widget is guaranteed to stay around until the 
            ///< "custom-widget-apply" signal is emitted on the operation. Then you can read 
            ///< out any information you need from the widgets.

            const CustomWidgetApplySignalProxy signal_custom_widget_apply();
            ///< Emitted right before "begin-print" if you added a custom widget in the 
            ///< create-custom-widget handler. When you get this signal you should read 
            ///< the information from the custom widgets, as the widgets are not 
            ///< guaraneed to be around at a later time.

            const DoneSignalProxy signal_done();
            ///< Emitted when the print operation run has finished doing everything 
            ///< required for printing. result gives you information about what 
            ///< happened during the run. If result is PRINT_OPERATION_RESULT_ERROR 
            ///< then you can call get_error() for more information.
            ///< If you enabled print status tracking then is_finished() may still 
            ///< return false after "done" was emitted.
            
            const DrawPageSignalProxy signal_draw_page();
            ///< Emitted for every page that is printed. The signal handler must render 
            ///< the page_nr's page onto the cairo context obtained from context 
            ///< using PrintContext::get_cairo_context().
            ///< Use set_use_full_page() and set_unit() before starting the print 
            ///< operation to set up the transformation of the cairo context according to your needs.

            const EndPrintSignalProxy signal_end_print();
            ///< Emitted after all pages have been rendered. A handler for this signal 
            ///< can clean up any resources that have been allocated in the "begin-print" handler.

            const PaginateSignalProxy signal_paginate();
            ///< Emitted after the "begin-print" signal, but before the actual rendering starts. 
            ///< It keeps getting emitted until it returns false.
            ///<
            ///< The paginate signal is intended to be used for paginating the document in small 
            ///< chunks, to avoid blocking the user interface for a long time. The signal handler 
            ///< should update the number of pages using set_n_pages(), and return tru if the 
            ///< document has been completely paginated.
            ///< 
            ///< If you don't need to do pagination in chunks, you can simply do it all in the 
            ///< begin-print handler, and set the number of pages from there.

            const PreviewSignalProxy signal_preview();
            ///< Gets emitted when a preview is requested from the native dialog.
            ///< The default handler for this signal uses an external viewer application to preview.
            ///< To implement a custom print preview, an application must return true from its 
            ///< handler for this signal. In order to use the provided context for the preview 
            ///< implementation, it must be given a suitable cairo context with 
            ///< PrintContext::set_cairo_context().

            ///< The custom preview implementation can use PrintOperationPreview::is_selected() 
            ///< and PrintOperationPreview::render_page() to find pages which are selected for 
            ///< print and render them. The preview must be finished by calling PrintOperationPreview::end_preview() 
            ///< (typically in response to the user clicking a close button).
    
            const RequestPageSetupSignalProxy signal_request_page();
            ///< Emitted once for every page that is printed, to give the application a 
            ///< chance to modify the page setup. Any changes done to setup will be in force 
            ///< only for printing this page.

            const StatusChangedSignalProxy signal_status_changed();
            ///< Emitted at between the various phases of the print operation. See PrintStatus 
            ///< for the phases that are being discriminated. Use get_status() to find out the
            ///< current status.

            /// @}
        };

        inline GtkPrintOperation* PrintOperation::gtk_print_operation() const
        {
            return reinterpret_cast<GtkPrintOperation*>(instance_);
        }

        inline PrintOperation::operator GtkPrintOperation* () const
        {
            return this ? gtk_print_operation() : NULL;
        }
        
    }
}

#endif
