/*  xfce4-terminal
 *
 *  Copyright (C) 2004 Eduard Roccatello (eduard@xfce.org)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <vte/vte.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4mcs/mcs-client.h>
#include <settings/settings.h>

#include <callbacks.h>
#include <xfce4-terminal.h>
#include <accelerators.h>
#include <globals.h>
#include "inline-icon.h"

/* Use the item factory defined above to generate the menu, with accels. 
 * As a result of using the item factory, all callbacks receive a pointer
 * to the calling Terminal as callback_data.   */
static GtkWidget *get_menubar_menu (Terminal *t)
{
	t->item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", NULL);
	gtk_item_factory_create_items(t->item_factory, nmenu_items, menu_items, t);
	return gtk_item_factory_get_widget(t->item_factory, "<main>");
}

/* Build up a new terminal from given data */
Shell *create_terminal(const gchar *shell, const gchar *pwd, const gchar *label, gpointer data)
{
	Shell *s = g_new(Shell, 1);
	s->vte = GTK_WIDGET(vte_terminal_new());
    s->parent = data;
    if (settings && settings->font)
        vte_terminal_set_font_from_string (VTE_TERMINAL(s->vte), settings->font);
    else
	   vte_terminal_set_font_from_string (VTE_TERMINAL(s->vte), DEFAULT_TERMINAL_FONTNAME);
	s->label = (label != NULL ? gtk_label_new(label) : NULL);

	vte_terminal_fork_command(VTE_TERMINAL(s->vte), shell, NULL, NULL, pwd, FALSE, FALSE, FALSE);
	g_signal_connect (s->vte, "child-exited", G_CALLBACK(cb_childexit), (gpointer)s);
	g_signal_connect (s->vte, "window-title-changed", G_CALLBACK(cb_title_changed), (gpointer)s);
	g_signal_connect (s->vte, "button-press-event", G_CALLBACK(cb_vte_button_press), (gpointer)data);

	gtk_widget_show(s->vte);
	if (s->label != NULL) gtk_widget_show(s->label);

    while (gtk_events_pending ())
	  gtk_main_iteration ();

	return s;
}

/* Build up the main interface */
Terminal *create_interface(void)
{
	Terminal *t;
	Shell *s;

	t = g_new(Terminal, 1);
	t->terms = g_ptr_array_new();
    t->menubar = NULL;

    /* Settings fetch */
    settings = g_new(Settings, 1);
    mcs_client = settings_init(t);
	
	t->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	g_signal_connect(t->window, "delete_event", gtk_main_quit, NULL);
	gtk_window_set_title(GTK_WINDOW(t->window), "Xfce4 Terminal");
	gtk_window_set_icon(GTK_WINDOW(t->window), xfce_inline_icon_at_size (default_icon_data_48_48, 48, 48));

	/* Setup the vbox */
	t->vbox = gtk_vbox_new(FALSE, 1);
	gtk_container_add(GTK_CONTAINER(t->window),t->vbox);
	gtk_widget_show (t->vbox);
	
	/* Put the menubar in */
	t->menubar = get_menubar_menu(t);
	gtk_box_pack_start(GTK_BOX(t->vbox), t->menubar, FALSE, TRUE, 0);
    if (settings->menu == TOGGLE_PRESSED)
    	gtk_widget_show (t->menubar);
	
	t->notebook = gtk_notebook_new();
    gtk_notebook_set_scrollable (GTK_NOTEBOOK(t->notebook), TRUE);
	/* Initially we have only a shell opened so tabs are futile */
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK(t->notebook), FALSE);
	g_signal_connect (t->notebook, "switch-page", G_CALLBACK(cb_switch_notebook_tab), (gpointer)t);
	gtk_box_pack_start(GTK_BOX(t->vbox), t->notebook, TRUE, TRUE, 0);
	gtk_widget_show (t->notebook);
	
	s = create_terminal(getenv("SHELL"), getenv("PWD"), NULL, (gpointer)t);
	gtk_notebook_append_page (GTK_NOTEBOOK(t->notebook), s->vte, s->label);
    s->page = gtk_notebook_get_n_pages (GTK_NOTEBOOK(t->notebook)) - 1;
	g_ptr_array_add (t->terms, s);

    init_terminal_accelerators(t);

	/* menu accelerator */
    if (settings->accels == TOGGLE_PRESSED) {
    	gtk_settings_set_string_property(gtk_settings_get_default (), "gtk-menu-bar-accel", 
            DEFAULT_ACCEL, "xfce-terminal");
    }
    else {
    	gtk_settings_set_string_property(gtk_settings_get_default (), "gtk-menu-bar-accel", 
            VOID_ACCEL, "xfce-terminal");
    }
    
    /* This is not the best way so: MUST FIX */
    g_object_set (G_OBJECT(t->window), "allow-shrink", TRUE, NULL);

	gtk_widget_show (t->window);
	return t;
}


void *(_font_updater) (gpointer data, gpointer user_data)
{
    Shell *s = data;
    g_return_if_fail (s);
    
    if (settings && settings->font)
        vte_terminal_set_font_from_string (VTE_TERMINAL(s->vte), settings->font);
    else
	    vte_terminal_set_font_from_string (VTE_TERMINAL(s->vte), DEFAULT_TERMINAL_FONTNAME);

}

void update_settings(Terminal *t)
{
    guint i;
    g_return_if_fail (t);
    g_return_if_fail (t->terms);

    for (i = 0; i < t->terms->len; i++)
        (*_font_updater) (t->terms->pdata[i], NULL);
    /* Not supported in Glib < 2.3.2: g_ptr_array_foreach (t->terms, (GFunc)_font_updater, NULL);*/

    if (t->menubar && GTK_IS_WIDGET(t->menubar))
    {
        if (settings->menu == TOGGLE_PRESSED)
            gtk_widget_show (t->menubar);
        else
            gtk_widget_hide (t->menubar);

        if (settings->accels == TOGGLE_PRESSED) {
        	gtk_settings_set_string_property(gtk_settings_get_default (), "gtk-menu-bar-accel", 
                DEFAULT_ACCEL, "xfce-terminal");
        }
        else {
        	gtk_settings_set_string_property(gtk_settings_get_default (), "gtk-menu-bar-accel", 
                VOID_ACCEL, "xfce-terminal");
        }
    }
}

GdkFilterReturn
mcs_client_event_filter (GdkXEvent * xevent, GdkEvent * event, gpointer data)
{
    if (mcs_client_process_event (mcs_client, (XEvent *) xevent))
	   return GDK_FILTER_REMOVE;
	return GDK_FILTER_CONTINUE;
}

static void
cb_mcs_watch (Window window, Bool is_start, long mask, void *data)
{
    GdkWindow *gdkwin;

    gdkwin = gdk_window_lookup (window);

    if (is_start)
	gdk_window_add_filter (gdkwin, mcs_client_event_filter, NULL);
    else
	gdk_window_remove_filter (gdkwin, mcs_client_event_filter, NULL);

}

/* event handling */
static void
cb_mcs_notify (const char *name, const char *channel_name,
	   McsAction action, McsSetting * setting, void *data)
{

    if (g_ascii_strcasecmp (CHANNEL, channel_name))
	return;

    switch (action)
    {
    	case MCS_ACTION_NEW:
    	    /* fall through */
    	case MCS_ACTION_CHANGED:
            if ((g_ascii_strcasecmp (CHANNEL_TERMINAL_FONTNAME, name)==0) && (setting->type == MCS_TYPE_STRING))
                settings->font = g_strdup(setting->data.v_string);
            if (g_ascii_strcasecmp (CHANNEL_TERMINAL_SHOWMENU, name)==0)
                settings->menu = setting->data.v_int;
            if (g_ascii_strcasecmp (CHANNEL_TERMINAL_ACCELS, name)==0)
                settings->accels = setting->data.v_int;
    	    break;
    	case MCS_ACTION_DELETED:
    	    /* We don't use this now. Perhaps revert to default? */
    	    break;
    }

    if (data)
        update_settings(data);

}

McsClient *
settings_init(Terminal *t)
{
	Display *xdpy;
	gint xscreen;
	
	g_return_val_if_fail(mcs_client == NULL, mcs_client);
	
	xdpy = GDK_DISPLAY();
	xscreen = gdk_screen_get_number(gdk_display_get_default_screen(gdk_display_get_default()));
	
	if(!mcs_client_check_manager(xdpy, xscreen, "xfce-mcs-manager")) {
		g_critical("%s: Unable to start settings manager", PACKAGE);
		return NULL;
	}
	
    mcs_client = mcs_client_new(xdpy, xscreen, cb_mcs_notify, cb_mcs_watch, (gpointer)t);
	if(!mcs_client) {
		g_critical("%s: Unable to connect to settings manager\n", PACKAGE);
		return NULL;
	}
	mcs_client_add_channel(mcs_client, CHANNEL);
	
	return mcs_client;
}

/* Main function */
gint
main (gint argc, gchar **argv)
{
	Terminal *t = NULL;

	gtk_init(&argc, &argv);

	t = create_interface();
	gtk_main();

	if(mcs_client) {
		mcs_client_destroy(mcs_client);
		mcs_client = NULL;
	}

	return 0;
}
