/* $Id: xfce-exec.c,v 1.6 2004/06/15 22:13:47 olivier Exp $
 * 
 * Copyright (C) 2004 Jasper Huijsmans <jasper@xfce.org>
 * startup notification added by Olivier Fourdan based on gnome-desktop
 * developed by Elliot Lee <sopwith@redhat.com> and Sid Vicious
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#ifdef HAVE_STDDEF_H
#include <stddef.h>
#endif

#include <gdk/gdk.h>

#ifdef HAVE_LIBSTARTUP_NOTIFICATION
#define SN_API_NOT_YET_FROZEN
#include <libsn/sn.h>
#include <gdk/gdkx.h>
#define STARTUP_TIMEOUT (30 /* seconds */ * 1000)
#endif

#include "xfce-exec.h"

extern char **environ;

#ifdef HAVE_LIBSTARTUP_NOTIFICATION

typedef struct
{
    GSList *contexts;
    guint timeout_id;
}
StartupTimeoutData;

static StartupTimeoutData *startup_timeout_data = NULL;

static gboolean atexit_registered = FALSE;

static void
sn_error_trap_push (SnDisplay * display, Display * xdisplay)
{
    gdk_error_trap_push ();
}

static void
sn_error_trap_pop (SnDisplay * display, Display * xdisplay)
{
    gdk_error_trap_pop ();
}

static gchar **
make_spawn_environment_for_sn_context (SnLauncherContext *
				       sn_context, char **envp)
{
    gchar **retval = NULL;
    int i, j;
    int desktop_startup_id_len;

    if (envp == NULL)
    {
	envp = environ;
    }
    for (i = 0; envp[i]; i++);

    retval = g_new (gchar *, i + 2);

    desktop_startup_id_len = strlen ("DESKTOP_STARTUP_ID");

    for (i = 0, j = 0; envp[i]; i++)
    {
	if (strncmp (envp[i], "DESKTOP_STARTUP_ID", desktop_startup_id_len) !=
	    0)
	{
	    retval[j] = g_strdup (envp[i]);
	    ++j;
	}
    }

    retval[j] =
	g_strdup_printf ("DESKTOP_STARTUP_ID=%s",
			 sn_launcher_context_get_startup_id (sn_context));
    ++j;
    retval[j] = NULL;

    return retval;
}

static gboolean
startup_timeout (void *data)
{
    StartupTimeoutData *std = data;
    GSList *tmp;
    GTimeVal now;
    int min_timeout;

    min_timeout = STARTUP_TIMEOUT;

    g_get_current_time (&now);

    tmp = std->contexts;
    while (tmp != NULL)
    {
	SnLauncherContext *sn_context = tmp->data;
	GSList *next = tmp->next;
	long tv_sec, tv_usec;
	double elapsed;

	sn_launcher_context_get_last_active_time (sn_context, &tv_sec,
						  &tv_usec);

	elapsed =
	    ((((double) now.tv_sec - tv_sec) * G_USEC_PER_SEC +
	      (now.tv_usec - tv_usec))) / 1000.0;

	if (elapsed >= STARTUP_TIMEOUT)
	{
	    std->contexts = g_slist_remove (std->contexts, sn_context);
	    sn_launcher_context_complete (sn_context);
	    sn_launcher_context_unref (sn_context);
	}
	else
	{
	    min_timeout = MIN (min_timeout, (STARTUP_TIMEOUT - elapsed));
	}

	tmp = next;
    }

    if (std->contexts == NULL)
    {
	std->timeout_id = 0;
    }
    else
    {
	std->timeout_id = g_timeout_add (min_timeout, startup_timeout, std);
    }

    return FALSE;
}

static void
add_startup_timeout (SnLauncherContext * sn_context)
{
    if (startup_timeout_data == NULL)
    {
	startup_timeout_data = g_new (StartupTimeoutData, 1);
	startup_timeout_data->contexts = NULL;
	startup_timeout_data->timeout_id = 0;
    }

    sn_launcher_context_ref (sn_context);
    startup_timeout_data->contexts =
	g_slist_prepend (startup_timeout_data->contexts, sn_context);

    if (startup_timeout_data->timeout_id == 0)
    {
	startup_timeout_data->timeout_id =
	    g_timeout_add (STARTUP_TIMEOUT, startup_timeout,
			   startup_timeout_data);
    }
}

static void
free_startup_timeout (void)
{
    StartupTimeoutData *std = startup_timeout_data;

    if (!std)
    {
	/* No startup notification used, return silently */
	return;
    }

    g_slist_foreach (std->contexts, (GFunc) sn_launcher_context_unref, NULL);
    g_slist_free (std->contexts);

    if (std->timeout_id != 0)
    {
	g_source_remove (std->timeout_id);
	std->timeout_id = 0;
    }

    g_free (std);
    startup_timeout_data = NULL;

    atexit_registered = FALSE;
}

#endif /* HAVE_LIBSTARTUP_NOTIFICATION */

/**
 * xfce_exec
 * @cmd         : command line to run
 * @in_terminal : whether to run @cmd in a terminal
 * @use_sn      : whether to use startup notification
 * @error       : location for a GError or NULL
 *
 * Use #xfce_exec_with_envp if you want to specify the environment.
 * 
 * Returns: TRUE on success, FALSE on failure. 
 */
gboolean xfce_exec (const char *cmd, 
		    gboolean in_terminal, 
		    gboolean use_sn, 
		    GError **error)
{
    return xfce_exec_with_envp (cmd, in_terminal, use_sn, error, environ);
}

/**
 * xfce_exec_with_envp
 * @cmd         : command line to run
 * @in_terminal : whether to run @cmd in a terminal
 * @use_sn      : whether to use startup notification
 * @error       : location for a GError or NULL
 * @envp	: the environment
 *
 * See also #xfce_exec.
 *
 * Returns: TRUE on success, FALSE on failure. 
 */
gboolean
xfce_exec_with_envp (const char *cmd, 
		     gboolean in_terminal, 
		     gboolean use_sn,
		     GError ** error,
		     char **envp)
{
    gchar *execute = NULL;
    gchar **free_envp = NULL;
    gchar **argv = NULL;
    gboolean success;
    
#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    SnLauncherContext *sn_context = NULL;
    SnDisplay *sn_display = NULL;

    if (!atexit_registered)
    {
	g_atexit (free_startup_timeout);
	atexit_registered = TRUE;
    }
#endif

    if (g_path_is_absolute (cmd) && g_file_test (cmd, G_FILE_TEST_IS_DIR))
    {
	if (in_terminal)
	{
	    execute = g_strconcat ("xfterm4 ", cmd, NULL);
	}
	else
	{
	    execute = g_strconcat ("xftree4 ", cmd, NULL);
	}
    }
    else if (in_terminal)
    {
	execute = g_strconcat ("xfterm4 -e ", cmd, NULL);
    }
    else
    {
	execute = g_strdup (cmd);
    }

    if (!g_shell_parse_argv (execute, NULL, &argv, error))
    {
	g_free (execute);

	return FALSE;
    }

    free_envp = NULL;

#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    if (use_sn)
    {
	sn_display = sn_display_new (gdk_display, sn_error_trap_push,
				     sn_error_trap_pop);

	if (sn_display != NULL)
	{
	    sn_context = 
		sn_launcher_context_new (sn_display,
					 DefaultScreen (gdk_display));
	    if ((sn_context != NULL)
		&& !sn_launcher_context_get_initiated (sn_context))
	    {
		sn_launcher_context_set_binary_name (sn_context, execute);
		sn_launcher_context_initiate (sn_context,
					      g_get_prgname () ?
					      g_get_prgname () : "unknown",
					      argv[0], CurrentTime);
		free_envp =
		    make_spawn_environment_for_sn_context (sn_context, envp);
	    }
	}
    }
#endif

    g_free (execute);

    success = g_spawn_async_with_pipes (NULL, 
	    				argv, free_envp ? free_envp : envp,
					G_SPAWN_SEARCH_PATH, 
					NULL, NULL, NULL,
					NULL, NULL, NULL,
					error);

#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    if (use_sn && success)
    {
	if (sn_context != NULL)
	{
	    if (!success)
	    {
		sn_launcher_context_complete (sn_context); /* end sequence */
	    }
	    else
	    {
		add_startup_timeout (sn_context);
		sn_launcher_context_unref (sn_context);
	    }
	}
	if (free_envp)
	{
	    g_strfreev (free_envp);
	    free_envp = NULL;
	}
	if (sn_display)
	{
	    sn_display_unref (sn_display);
	}
    }
#endif /* HAVE_LIBSTARTUP_NOTIFICATION */

    g_strfreev (argv);

    return success;
}
