/* 
 * xfce-appmenuitem - a GtkImageMenuItem subclass that handles menu
 * items that are intended to represent applications
 *
 * Copyright (c) 2004 Brian Tarricone <bjt23@cornell.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtkaccellabel.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtkversion.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/icons.h>
#include <libxfcegui4/xfce-appmenuitem.h>
#include <libxfcegui4/xfce-internals.h>

static void xfce_app_menu_item_class_init(XfceAppMenuItemClass *klass);
static void xfce_app_menu_item_init(XfceAppMenuItem *app_menu_item);
static void xfce_app_menu_item_set_property(GObject *object,
                                            guint prop_id,
                                            const GValue *value,
                                            GParamSpec *pspec);
static void xfce_app_menu_item_get_property(GObject *object,
                                            guint prop_id,
                                            GValue *value,
                                            GParamSpec *pspec);
static void xfce_app_menu_item_destroy(GtkObject *object);
static void xfce_app_menu_item_finalize(GObject *object);

static gint _calc_icon_size();
static void _ensure_icon_theme_instance();
static void _command_activate_cb(GtkMenuItem *menu_item,
                                 gpointer user_data);

enum {
	PROP_ZERO = 0,
	PROP_TERM,
	PROP_CMD,
	PROP_ICON
};

struct _XfceAppMenuItemPriv {
	gboolean destroyed;
	
	gchar *name;
	gchar *command;
	gboolean needs_term;
	gboolean snotify;
	gchar *icon_name;
};

static GtkImageMenuItemClass *parent_class = NULL;
static guint _xfce_app_menu_item_icon_size = 24;

static void
_command_activate_cb(GtkMenuItem *menu_item, gpointer user_data)
{
	XfceAppMenuItem *app_menu_item = XFCE_APP_MENU_ITEM(menu_item);
	GdkScreen *gscreen;
	gchar *dpy_name = NULL, *new_dpy_name, *p, *q;
	const gchar *old_display_env;
	
	g_return_if_fail(app_menu_item && app_menu_item->priv->command);
	
	old_display_env = g_getenv("DISPLAY");
	
	gscreen = gtk_widget_get_screen(GTK_WIDGET(menu_item));
	dpy_name = g_strdup(gdk_display_get_name(gdk_display_get_default()));
	p = g_strrstr(dpy_name, ":");
	q = g_strrstr(dpy_name, ".");
	if(p && q && p < q)
		*q = 0;
	if(p) {
		new_dpy_name = g_strdup_printf("%s.%d", dpy_name,
				GDK_SCREEN_XNUMBER(gscreen));
		xfce_setenv("DISPLAY", new_dpy_name, TRUE);
		g_free(new_dpy_name);
	}
	if(dpy_name)
		g_free(dpy_name);
	
	if(!xfce_exec(app_menu_item->priv->command,	app_menu_item->priv->needs_term,
			app_menu_item->priv->snotify, NULL))
	{
		g_warning("XfceAppMenuItem: unable to spawn %s\n",
				app_menu_item->priv->command);
	}
	
	xfce_setenv("DISPLAY", old_display_env, TRUE);
}

GType
xfce_app_menu_item_get_type()
{
	static GType app_menu_item_type = 0;
	
	if(!app_menu_item_type) {
		static const GTypeInfo app_menu_item_info = {
			sizeof(XfceAppMenuItemClass),
			NULL,  /* base_init */
			NULL,  /* base_finalize */
			(GClassInitFunc)xfce_app_menu_item_class_init,
			NULL,  /* class_finalize */
			NULL,  /* class_data */
			sizeof(XfceAppMenuItem),
			0,  /* n_preallocs */
			(GInstanceInitFunc)xfce_app_menu_item_init,
		};
		
		app_menu_item_type = g_type_register_static(GTK_TYPE_IMAGE_MENU_ITEM,
				"XfceAppMenuItem", &app_menu_item_info, 0);
	}
	
	return app_menu_item_type;
}

static void
xfce_app_menu_item_class_init(XfceAppMenuItemClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	/* be sure to initialize the library's i18n support first */
	_xfce_i18n_init ();
	
	gobject_class = (GObjectClass *)klass;
	object_class = (GtkObjectClass *)klass;
	
	parent_class = g_type_class_peek_parent(klass);
	
	object_class->destroy = xfce_app_menu_item_destroy;
	
	gobject_class->finalize = xfce_app_menu_item_finalize;
	gobject_class->set_property = xfce_app_menu_item_set_property;
	gobject_class->get_property = xfce_app_menu_item_get_property;
	
	g_object_class_install_property(gobject_class, PROP_TERM,
			g_param_spec_boolean("needs_term", _("Needs terminal"), 
					_("Whether or not the command needs a terminal to execute"),
					FALSE, G_PARAM_READABLE | G_PARAM_WRITABLE));
	
	g_object_class_install_property(gobject_class, PROP_CMD,
			g_param_spec_string("command", _("Command"),
					_("The command to run when the item is clicked"),
					NULL, G_PARAM_READABLE | G_PARAM_WRITABLE));
	
	g_object_class_install_property(gobject_class, PROP_ICON,
			g_param_spec_string("icon_name", _("Icon name"),
					_("The name of the themed icon to display next to the item"),
					NULL, G_PARAM_READABLE | G_PARAM_WRITABLE));
}

static void
xfce_app_menu_item_init(XfceAppMenuItem *app_menu_item)
{
	app_menu_item->priv = g_new0(XfceAppMenuItemPriv, 1);
	app_menu_item->priv->destroyed = FALSE;
}

static void
xfce_app_menu_item_set_property(GObject *object, guint prop_id,
		const GValue *value, GParamSpec *pspec)
{
	XfceAppMenuItem *app_menu_item = XFCE_APP_MENU_ITEM(object);
	
	switch(prop_id) {
		case PROP_TERM:
			{
				gboolean nterm = g_value_get_boolean(value);
				xfce_app_menu_item_set_needs_term(app_menu_item, nterm);
			}
			break;
		case PROP_CMD:
			{
				const gchar *cmd = g_value_get_string(value);
				xfce_app_menu_item_set_command(app_menu_item, (gchar *)cmd);
			}
			break;
		case PROP_ICON:
			{
				const gchar *icon = g_value_get_string(value);
				xfce_app_menu_item_set_icon_name(app_menu_item, (gchar *)icon);
			}
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
xfce_app_menu_item_get_property(GObject *object, guint prop_id, GValue *value,
		GParamSpec *pspec)
{
	XfceAppMenuItem *app_menu_item = XFCE_APP_MENU_ITEM(object);
	
	switch(prop_id) {
		case PROP_TERM:
			g_value_set_boolean(value, app_menu_item->priv->needs_term);
			break;
		case PROP_CMD:
			g_value_set_string(value, app_menu_item->priv->command);
			break;
		case PROP_ICON:
			g_value_set_string(value, app_menu_item->priv->icon_name);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
xfce_app_menu_item_toggle_size_request(GtkMenuItem *menu_item, gint *requisition)
{
	XfceAppMenuItem *app_menu_item = XFCE_APP_MENU_ITEM(menu_item);
	
	if(app_menu_item->image_menu_item.image)
		*requisition = app_menu_item->image_menu_item.image->requisition.width;
	else
		*requisition = 0;
}

static void
xfce_app_menu_item_destroy(GtkObject *object)
{
	XfceAppMenuItem *app_menu_item = XFCE_APP_MENU_ITEM(object);
	
	g_return_if_fail(app_menu_item != NULL);
	
	if(app_menu_item->priv->destroyed)
		return;
	
	app_menu_item->priv->destroyed = TRUE;

	if(app_menu_item->priv->name)
		g_free(app_menu_item->priv->name);
	if(app_menu_item->priv->command)
		g_free(app_menu_item->priv->command);
	if(app_menu_item->priv->icon_name)
		g_free(app_menu_item->priv->icon_name);
	
	(*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}

static void
xfce_app_menu_item_finalize(GObject *object)
{
	XfceAppMenuItem *app_menu_item = XFCE_APP_MENU_ITEM(object);
	
	g_return_if_fail(app_menu_item != NULL);
	
	if(!app_menu_item->priv->destroyed)
		xfce_app_menu_item_destroy(GTK_OBJECT(object));
	
	g_free(app_menu_item->priv);
	
	(*G_OBJECT_CLASS(parent_class)->finalize)(object);
}

/**
 * xfce_app_menu_item_new:
 * @returns: A new #XfceAppMenuItem.
 *
 * Creates a new #XfceAppMenuItem with an empty label.
 *
 * Since 4.1
 **/
GtkWidget *
xfce_app_menu_item_new()
{
	XfceAppMenuItem *app_menu_item = g_object_new(XFCE_TYPE_APP_MENU_ITEM, NULL);
	
	g_signal_connect(G_OBJECT(app_menu_item), "activate",
			G_CALLBACK(_command_activate_cb), NULL);
	
	return GTK_WIDGET(app_menu_item);
}

/**
 * xfce_app_menu_item_new_with_label:
 * @label: The text of the menu item.
 * @returns: A new #XfceAppMenuItem.
 *
 * Creates a new #XfceAppMenuItem containing a label.
 *
 * Since 4.1
 **/
GtkWidget *
xfce_app_menu_item_new_with_label(const gchar *label)
{
	XfceAppMenuItem *app_menu_item;
	GtkWidget *accel_label;
	
	app_menu_item = g_object_new(XFCE_TYPE_APP_MENU_ITEM, NULL);
	
	app_menu_item->priv->name = g_strdup(label);
	
	accel_label = gtk_accel_label_new(app_menu_item->priv->name);
	gtk_misc_set_alignment(GTK_MISC(accel_label), 0.0, 0.5);

	gtk_container_add(GTK_CONTAINER(app_menu_item), accel_label);
	gtk_accel_label_set_accel_widget(GTK_ACCEL_LABEL(accel_label),
			GTK_WIDGET(app_menu_item));
	gtk_widget_show(accel_label);
	
	g_signal_connect(G_OBJECT(app_menu_item), "activate",
			G_CALLBACK(_command_activate_cb), NULL);

	return GTK_WIDGET(app_menu_item);
}

/**
 * xfce_app_menu_item_new_with_mnemonic:
 * @label: The text of the menu item, with an underscore in front of the
 *         mnemonic character.
 * @returns: A new #XfceAppMenuItem.
 *
 * Creates a new #XfceAppMenuItem containing a label. The label
 * will be created using gtk_label_new_with_mnemonic(), so underscores
 * in @label indicate the mnemonic for the menu item.
 *
 * Since 4.1
 **/
GtkWidget *
xfce_app_menu_item_new_with_mnemonic(const gchar *label)
{
	XfceAppMenuItem *app_menu_item;
	GtkWidget *accel_label;
	
	app_menu_item = g_object_new(XFCE_TYPE_APP_MENU_ITEM, NULL);
	
	app_menu_item->priv->name = g_strdup(label);
	
	accel_label = g_object_new(GTK_TYPE_ACCEL_LABEL, NULL);
	gtk_label_set_text_with_mnemonic(GTK_LABEL(accel_label),
			app_menu_item->priv->name);
	gtk_misc_set_alignment(GTK_MISC(accel_label), 0.0, 0.5);

	gtk_container_add(GTK_CONTAINER(app_menu_item), accel_label);
	gtk_accel_label_set_accel_widget(GTK_ACCEL_LABEL(accel_label),
			GTK_WIDGET(app_menu_item));
	gtk_widget_show(accel_label);

	g_signal_connect(G_OBJECT(app_menu_item), "activate",
			G_CALLBACK(_command_activate_cb), NULL);
	
	return GTK_WIDGET(app_menu_item);
}

/**
 * xfce_app_menu_item_new_with_command:
 * @label: The text of the menu item.
 * @command: The command associated with the menu item.
 * @returns: A new #XfceAppMenuItem.
 *
 * Creates a new #XfceAppMenuItem containing a label. The item's @activate
 * signal will be connected such that @command will run when it is clicked.
 *
 * Since 4.1
 **/
GtkWidget *
xfce_app_menu_item_new_with_command(const gchar *label, const gchar *command)
{
	XfceAppMenuItem *app_menu_item;
	GtkWidget *accel_label;
	
	app_menu_item = g_object_new(XFCE_TYPE_APP_MENU_ITEM, NULL);
	
	app_menu_item->priv->name = g_strdup(label);
	
	accel_label = gtk_accel_label_new(app_menu_item->priv->name);
	gtk_misc_set_alignment(GTK_MISC(accel_label), 0.0, 0.5);

	gtk_container_add(GTK_CONTAINER(app_menu_item), accel_label);
	gtk_accel_label_set_accel_widget(GTK_ACCEL_LABEL(accel_label),
			GTK_WIDGET(app_menu_item));
	gtk_widget_show(accel_label);
	
	app_menu_item->priv->command = xfce_expand_variables(command, NULL);
	g_signal_connect(G_OBJECT(app_menu_item), "activate",
			G_CALLBACK(_command_activate_cb), NULL);
	
	return GTK_WIDGET(app_menu_item);
}

/**
 * xfce_app_menu_item_new_full:
 * @label: The text of the menu item.
 * @command: The command associated with the menu item.
 * @icon_filename: The filename of the icon.
 * @needs_term: TRUE if the application needs a terminal, FALSE if not.
 * @snotify: TRUE if the application supports startup notification, FALSE if
 *           not.
 * @returns: A new #XfceAppMenuItem.
 *
 * Single-function interface to create an #XfceAppMenuItem.  Has the effect of
 * calling xfce_app_menu_item_new_with_label() followed by all the
 * xfce_app_menu_item_set_*() functions.
 *
 * Since 4.1
 **/
GtkWidget *xfce_app_menu_item_new_full(const gchar *label, const gchar *command,
		const gchar *icon_filename, gboolean needs_term, gboolean snotify)
{
	XfceAppMenuItem *app_menu_item = NULL;
	
	if(!label)
		app_menu_item = XFCE_APP_MENU_ITEM(xfce_app_menu_item_new());
	else {
		app_menu_item = XFCE_APP_MENU_ITEM(xfce_app_menu_item_new_with_label(label));
		if(command)
			xfce_app_menu_item_set_command(app_menu_item, command);
		if(icon_filename)
			xfce_app_menu_item_set_icon_name(app_menu_item, icon_filename);
		if(needs_term)
			xfce_app_menu_item_set_needs_term(app_menu_item, needs_term);
		if(snotify)
			xfce_app_menu_item_set_startup_notification(app_menu_item, snotify);
	}
	
	return GTK_WIDGET(app_menu_item);
}

/**
 * xfce_app_menu_item_new_from_desktop_entry:
 * @entry: An #XfceDesktopEntry describing the menu item to create.
 * @show_icon: Sets whether or not to show an icon in the menu item.
 * @returns: A new #XfceAppMenuItem.
 *
 * Creates a new #XfceAppMenuItem using parameters from the application
 * specified in a #XfceDesktopEntry object. This has the effect of calling
 * xfce_app_menu_item_new_with_command(), xfce_app_menu_item_set_needs_term(),
 * xfce_app_menu_item_set_icon_name(), and
 * xfce_app_menu_item_set_startup_notification().
 *
 * Since 4.1
 **/
GtkWidget *
xfce_app_menu_item_new_from_desktop_entry(XfceDesktopEntry *entry,
		gboolean show_icon)
{
	XfceAppMenuItem *app_menu_item;
	GtkWidget *accel_label;
	gchar *name = NULL, *cmd = NULL, *icon = NULL, *snotify = NULL, *p;
	gint term = 0;
	
	g_return_val_if_fail(XFCE_IS_DESKTOP_ENTRY(entry), NULL);
	
	app_menu_item = g_object_new(XFCE_TYPE_APP_MENU_ITEM, NULL);
	
	if(!xfce_desktop_entry_get_string(entry, "Name", TRUE, &name)) {
		gchar *tmp, *tmp1;
		tmp = g_strdup(xfce_desktop_entry_get_file(entry));
		if((tmp1 = g_strrstr(tmp, ".desktop")))
			*tmp1 = 0;
		if((tmp1 = g_strrstr(tmp, "/")))
			tmp1++;
		else
			tmp1 = name;
		name = g_strdup(tmp1);
		g_free(tmp);
	}
	app_menu_item->priv->name = name;
	
	accel_label = gtk_accel_label_new(app_menu_item->priv->name);
	gtk_misc_set_alignment(GTK_MISC(accel_label), 0.0, 0.5);

	gtk_container_add(GTK_CONTAINER(app_menu_item), accel_label);
	gtk_accel_label_set_accel_widget(GTK_ACCEL_LABEL(accel_label),
			GTK_WIDGET(app_menu_item));
	gtk_widget_show(accel_label);
	
	xfce_desktop_entry_get_int(entry, "Terminal", &term);
	app_menu_item->priv->needs_term = (term != 0 ? TRUE : FALSE);
	
	if(xfce_desktop_entry_get_string(entry, "StartupNotify", TRUE, &snotify)) {
		if(!strcmp(snotify, "true"))
			app_menu_item->priv->snotify = TRUE;
		g_free(snotify);
	}
	
	
	xfce_desktop_entry_get_string(entry, "Exec", TRUE, &cmd);
	/* naive approach to sanitising these */
	if((p=strchr(cmd, '%'))) {
		if((p=strchr(cmd, ' ')))
			*p = 0;
	}	
	app_menu_item->priv->command = xfce_expand_variables(cmd, NULL);
	g_free(cmd);
	
	if(show_icon) {
		xfce_desktop_entry_get_string(entry, "Icon", TRUE, &icon);
		if(icon) {
			xfce_app_menu_item_set_icon_name(app_menu_item, icon);
			g_free(icon);
		}
	}
	
	g_signal_connect(G_OBJECT(app_menu_item), "activate",
			G_CALLBACK(_command_activate_cb), NULL);
	
	return GTK_WIDGET(app_menu_item);
}

/**
 * xfce_app_menu_item_set_name:
 * @app_menu_item: An #XfceAppMenuItem.
 * @name: The name of the menu item the menu item.
 *
 * Sets @name as the displayed name of the #XfceAppMenuItem.
 *
 * Since 4.1
 **/
void
xfce_app_menu_item_set_name(XfceAppMenuItem *app_menu_item, const gchar *name)
{
	g_return_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item));
	
	if(app_menu_item->priv->name)
		g_free(app_menu_item->priv->name);
	
	app_menu_item->priv->name = g_strdup(name);
}

/**
 * xfce_app_menu_item_set_icon_name:
 * @app_menu_item: an #XfceAppMenuItem.
 * @filename: The filename of the icon.
 *
 * Sets the icon of the #XfceAppMenuItem using the specified filename.  If
 * the filename doesn't have a full pathname, standard icon search paths
 * will be used.  If the filename doesn't have an extension, the best image
 * format found (if any) will be used.  If there is already an icon set, the
 * current one is freed, regardless if the icon is found or not.
 *
 * Since 4.1
 **/
void
xfce_app_menu_item_set_icon_name(XfceAppMenuItem *app_menu_item,
		const gchar *filename)
{
	GtkWidget *image;
	GdkPixbuf *pix;
	gchar *dot;
	gint w, h;
	
	g_return_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item));

	if(app_menu_item->priv->icon_name)
		g_free(app_menu_item->priv->icon_name);
	
	app_menu_item->priv->icon_name = g_strdup(filename);
	
	pix = xfce_load_themed_icon(app_menu_item->priv->icon_name,
			_xfce_app_menu_item_icon_size);
	if(pix) {
		w = gdk_pixbuf_get_width(pix);
		h = gdk_pixbuf_get_height(pix);
		if(w != _xfce_app_menu_item_icon_size || h != _xfce_app_menu_item_icon_size) {
			GdkPixbuf *tmp = gdk_pixbuf_scale_simple(pix,
					_xfce_app_menu_item_icon_size, _xfce_app_menu_item_icon_size,
					GDK_INTERP_BILINEAR);
			g_object_unref(G_OBJECT(pix));
			pix = tmp;
		}
		image = gtk_image_new_from_pixbuf(pix);
		g_object_unref(G_OBJECT(pix));
		gtk_widget_show(image);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(app_menu_item), image);
	} else
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(app_menu_item), NULL);
}

/**
 * xfce_app_menu_item_set_command:
 * @app_menu_item: An #XfceAppMenuItem.
 * @command: The command to associate with the menu item.
 *
 * Sets @command as the command run when the #XfceAppMenuItem is clicked.
 *
 * Since 4.1
 **/
void
xfce_app_menu_item_set_command(XfceAppMenuItem *app_menu_item,
		const gchar *command)
{
	g_return_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item));
	
	if(app_menu_item->priv->command)
		g_free(app_menu_item->priv->command);

	app_menu_item->priv->command = xfce_expand_variables(command, NULL);
}

/**
 * xfce_app_menu_item_set_needs_term:
 * @app_menu_item: An #XfceAppMenuItem.
 * @needs_term: TRUE if the application needs a terminal, FALSE if not.
 *
 * Sets whether or not the command executed by this #XfceAppMenuItem requires
 * a terminal window to run.
 *
 * Since 4.1
 **/
void
xfce_app_menu_item_set_needs_term(XfceAppMenuItem *app_menu_item,
		gboolean needs_term)
{
	g_return_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item));
	
	app_menu_item->priv->needs_term = needs_term;
}

/**
 * xfce_app_menu_item_set_startup_notification:
 * @app_menu_item: An #XfceAppMenuItem.
 * @snotify: TRUE if the application supports startup notification, FALSE if
 *           not.
 *
 * Sets whether or not the application supports startup notification.
 *
 * Since 4.1
 **/
void
xfce_app_menu_item_set_startup_notification(XfceAppMenuItem *app_menu_item,
		gboolean snotify)
{
	g_return_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item));
	
	app_menu_item->priv->snotify = snotify;
}

/**
 * xfce_app_menu_item_get_name:
 * @app_menu_item: An #XfceAppMenuItem.
 * @returns: A name/label string.
 *
 * Returns the current name/label set for the #XfceAppMenuItem, or NULL.
 *
 * Since 4.1
 **/
G_CONST_RETURN gchar *
xfce_app_menu_item_get_name(XfceAppMenuItem *app_menu_item)
{
	g_return_val_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item), NULL);
	
	return app_menu_item->priv->name;
}

/**
 * xfce_app_menu_item_get_icon_name:
 * @app_menu_item: An #XfceAppMenuItem.
 * @returns: An icon name string.
 *
 * Returns the current icon name set for the #XfceAppMenuItem, or NULL.
 *
 * Since 4.1
 **/
G_CONST_RETURN gchar *
xfce_app_menu_item_get_icon_name(XfceAppMenuItem *app_menu_item)
{
	g_return_val_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item), NULL);
	
	return app_menu_item->priv->name;
}

/**
 * xfce_app_menu_item_get_command:
 * @app_menu_item: An #XfceAppMenuItem.
 * @returns: A command string.
 *
 * Returns the current command set for the #XfceAppMenuItem, or NULL.
 *
 * Since 4.1
 **/
G_CONST_RETURN gchar *
xfce_app_menu_item_get_command(XfceAppMenuItem *app_menu_item)
{
	g_return_val_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item), NULL);
	
	return app_menu_item->priv->command;
}

/**
 * xfce_app_menu_item_get_needs_term:
 * @app_menu_item: An #XfceAppMenuItem.
 * @returns: TRUE if the item will spawn a terminal, FALSE if not.
 *
 * Checks whether or not the command executed by this #XfceAppMenuItem requires
 * a terminal window to run.
 *
 * Since 4.1
 **/
gboolean
xfce_app_menu_item_get_needs_term(XfceAppMenuItem *app_menu_item)
{
	g_return_val_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item), FALSE);
	
	return app_menu_item->priv->needs_term;
}

/**
 * xfce_app_menu_item_get_startup_notification:
 * @app_menu_item: An #XfceAppMenuItem.
 * @returns: TRUE if the item supports startup notification, FALSE if not.
 *
 * Checks whether or not the command executed by this #XfceAppMenuItem supports
 * startup notification.
 *
 * Since 4.1
 **/
gboolean
xfce_app_menu_item_get_startup_notification(XfceAppMenuItem *app_menu_item)
{
	g_return_val_if_fail(XFCE_IS_APP_MENU_ITEM(app_menu_item), FALSE);
	
	return app_menu_item->priv->snotify;
}

/**
 * xfce_app_menu_item_set_icon_size:
 * @icon_size: An icon size, in pixels.
 *
 * Sets the icon size that should be used when constructing #XfceAppMenuItems
 * with icons.  This is a @static method and does not need an instance of
 * #XfceAppMenuItem to exist before it is called.
 *
 * Since 4.1
 **/
void xfce_app_menu_item_set_icon_size(guint icon_size)
{
	_xfce_app_menu_item_icon_size = icon_size;
}

/**
 * xfce_app_menu_item_set_icon_theme_name:
 * @theme_name: A theme name.
 *
 * Sets the icon theme name that should be used when looking for icons for
 * #XfceAppMenuItems.  This is a @static method and does not need an instance
 * of #XfceAppMenuItem to exist before it is called.
 *
 * Since 4.1
 **/
void
xfce_app_menu_item_set_icon_theme_name(const gchar *theme_name)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gchar *origin = g_strdup_printf("%s:%d", __FILE__, __LINE__);
	gtk_settings_set_string_property(gtk_settings_get_default(),
			"gtk-icon-theme-name", theme_name, origin);
	g_free(origin);
#else
	xfce_set_icon_theme(theme_name);
#endif
}
