/*  xfce4
 *  Copyright (C) 2002 Jasper Huijsmans (huysmans@users.sourceforge.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk/gdk.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtkaspectframe.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>

#include "xfce_menubutton.h"

/* Forward declarations */

static void xfce_menubutton_class_init (XfceMenubuttonClass * class);
static void xfce_menubutton_init (XfceMenubutton * menubutton);
static void xfce_menubutton_finalize (GObject * object);
static void xfce_menubutton_size_allocate (GtkWidget * widget,
				           GtkAllocation * allocation);
static void xfce_menubutton_scale (XfceMenubutton * menubutton, int width,
				   int height);

/* Local data */

static GtkWidgetClass *parent_class = NULL;

GtkType
xfce_menubutton_get_type (void)
{
    static GtkType menubutton_type = 0;

    if (!menubutton_type)
    {
	static const GTypeInfo menubutton_info = {
	    sizeof (XfceMenubuttonClass),
	    NULL,		/* base_init */
	    NULL,		/* base_finalize */
	    (GClassInitFunc) xfce_menubutton_class_init,
	    NULL,		/* class_finalize */
	    NULL,		/* class_data */
	    sizeof (XfceMenubutton),
	    0,			/* n_preallocs */
	    (GInstanceInitFunc) xfce_menubutton_init
	};

	menubutton_type =
	    g_type_register_static (GTK_TYPE_BUTTON, "XfceMenubutton",
				    &menubutton_info, 0);
    }

    return menubutton_type;
}

static void
xfce_menubutton_class_init (XfceMenubuttonClass * class)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (class);
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass *) class;
    widget_class = (GtkWidgetClass *) class;
    gobject_class = G_OBJECT_CLASS (class);

    parent_class = gtk_type_class (gtk_button_get_type ());

    widget_class->size_allocate = xfce_menubutton_size_allocate;
    
    gobject_class->finalize = xfce_menubutton_finalize;
}

static void
xfce_menubutton_init (XfceMenubutton * menubutton)
{
    menubutton->hbox = gtk_hbox_new (FALSE, 10);
    gtk_widget_show (menubutton->hbox);
    gtk_container_add (GTK_CONTAINER (menubutton), menubutton->hbox);

    menubutton->image = gtk_image_new ();
    gtk_widget_show (menubutton->image);
    gtk_box_pack_start (GTK_BOX (menubutton->hbox), menubutton->image, FALSE,
			TRUE, 0);

    menubutton->pb = NULL;
    menubutton->scaled = FALSE;
    menubutton->icon_size = -1;

    menubutton->label = gtk_label_new (NULL);
    gtk_misc_set_alignment (GTK_MISC (menubutton->label), 0, 0.5);
    gtk_widget_show (menubutton->label);
    gtk_box_pack_start (GTK_BOX (menubutton->hbox), menubutton->label, TRUE,
			TRUE, 0);
}

GtkWidget *
xfce_menubutton_new (const char *text)
{
    GtkWidget *button;

    button = GTK_WIDGET (g_object_new (xfce_menubutton_get_type (), NULL));

    xfce_menubutton_set_text (XFCE_MENUBUTTON (button), text);

    return button;
}

GtkWidget *
xfce_menubutton_new_with_pixbuf (const char *text, GdkPixbuf * pixbuf)
{
    GtkWidget *button;

    button = xfce_menubutton_new (text);
    xfce_menubutton_set_pixbuf (XFCE_MENUBUTTON (button), pixbuf);

    return button;
}

GtkWidget *
xfce_menubutton_new_with_stock_icon (const char *text, const char *stock)
{
    GtkWidget *button;

    button = xfce_menubutton_new (text);
    xfce_menubutton_set_stock_icon (XFCE_MENUBUTTON (button), stock);

    return button;
}

void
xfce_menubutton_set_text (XfceMenubutton * menubutton, const char *text)
{
    g_return_if_fail (menubutton != NULL);
    g_return_if_fail (XFCE_IS_MENUBUTTON (menubutton));

    gtk_label_set_text (GTK_LABEL (menubutton->label), text);
}

void
xfce_menubutton_set_pixbuf (XfceMenubutton * menubutton, GdkPixbuf * pixbuf)
{
    g_return_if_fail (menubutton != NULL);
    g_return_if_fail (XFCE_IS_MENUBUTTON (menubutton));

    if (menubutton->pb)
	g_object_unref (menubutton->pb);

    menubutton->pb = pixbuf;
    g_object_ref (menubutton->pb);

    if (menubutton->icon_size > 0)
    {
        xfce_menubutton_scale (menubutton,
                               menubutton->icon_size, 
                               menubutton->icon_size);
    }
    else
    {
        gtk_image_set_from_pixbuf (GTK_IMAGE (menubutton->image), 
                                   menubutton->pb);
    }
}

void
xfce_menubutton_set_stock_icon (XfceMenubutton * menubutton,
				const char *stock)
{
    g_return_if_fail (menubutton != NULL);
    g_return_if_fail (XFCE_IS_MENUBUTTON (menubutton));

    if (menubutton->pb)
    {
	g_object_unref (menubutton->pb);
	menubutton->pb = NULL;
    }

    gtk_image_set_from_stock (GTK_IMAGE (menubutton->image), stock,
			      GTK_ICON_SIZE_MENU);
}

static void
xfce_menubutton_finalize (GObject * object)
{
    XfceMenubutton *menubutton;

    g_return_if_fail (object != NULL);
    g_return_if_fail (XFCE_IS_MENUBUTTON (object));

    menubutton = XFCE_MENUBUTTON (object);

    if (menubutton->pb)
    {
	g_object_unref (menubutton->pb);
	menubutton->pb = NULL;
    }

    G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void 
xfce_menubutton_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
    XfceMenubutton *menubutton;
    gint xthick, ythick;
    
    g_return_if_fail (widget != NULL);
    g_return_if_fail (XFCE_IS_MENUBUTTON (widget));
    g_return_if_fail (allocation != NULL);

    menubutton = XFCE_MENUBUTTON(widget);
    
    xthick = widget->style->xthickness + menubutton->hbox->style->xthickness;
    ythick = widget->style->ythickness + menubutton->hbox->style->ythickness;
    
    menubutton->icon_size = MIN(allocation->width - 2 * xthick, 
                                allocation->height - 2 * ythick);

    widget->allocation = *allocation;

#if 0
    g_message("size: %d x %d, scaled: %s", 
	      allocation->width,
	      allocation->height,
	      menubutton->scaled ? "yes" : "no");
#endif

    /* prevent race condition */
    if (!menubutton->scaled)
    {
	if (menubutton->pb)
	    xfce_menubutton_scale (menubutton,
                                   menubutton->icon_size, 
                                   menubutton->icon_size);
	menubutton->scaled = TRUE;
    }
    else
    {
	menubutton->scaled = FALSE;
    }
    
    parent_class->size_allocate (widget, allocation);
    
#if 0
    g_message("image size: %d x %d", 
	      menubutton->image->allocation.width,
	      menubutton->image->allocation.height);
#endif
}

static void
xfce_menubutton_scale (XfceMenubutton * menubutton, int width, int height)
{
    GdkPixbuf *tmp;
    float wratio, hratio;
    int w, h, pw, ph;

    g_return_if_fail (menubutton != NULL);
    g_return_if_fail (XFCE_IS_MENUBUTTON (menubutton));
    g_return_if_fail (menubutton->pb != NULL);
    g_return_if_fail (GDK_IS_PIXBUF (menubutton->pb));

    pw = gdk_pixbuf_get_width (menubutton->pb);
    ph = gdk_pixbuf_get_height (menubutton->pb);

    w = width <= 0 ? pw + 1 : width;
    h = height <= 0 ? ph + 1 : height;

    wratio = (float) pw / (float) w;
    hratio = (float) ph / (float) h;
/*
    g_printerr("w = %d, h = %d, pw = %d, ph = %d\n", w, h, pw, ph);
    g_printerr("wratio = %f, hratio = %f\n", wratio, hratio);
*/
    if (wratio <= 1.0 && hratio <= 1.0)
    {
	/* don't scale up -> ugly icons */
	gtk_image_set_from_pixbuf (GTK_IMAGE (menubutton->image),
				   menubutton->pb);
	return;
    }
    else if (wratio > hratio)
    {
	h = (float) ph / wratio;
    }
    else if (hratio > wratio)
    {
	w = (float) pw / hratio;
    }

    tmp = gdk_pixbuf_scale_simple (menubutton->pb, w, h, GDK_INTERP_BILINEAR);
    gtk_image_set_from_pixbuf (GTK_IMAGE (menubutton->image), tmp);
    g_object_unref (tmp);
}
