#!/usr/bin/ruby -w

require "gtk2"
require "singleton"

require "config"
require "compat18"
require "eataria2"

include Gtk

module Eat
end



class Eat::Settings

	include Singleton

	attr_reader :aria2
	attr_accessor :values

	def initialize()
		@conf_dir = ENV["XDG_CONFIG_HOME"]+"/eatmonkey/"
		@aria2 = Eat::Aria2Config.new(@conf_dir+"aria2.conf")
		@values = {}
		parse
	end

	private

	def default_values()
		# Download Directory
		# TODO use g_get_user_special_dir(G_USER_DIRECTORY_DOWNLOAD) or how not to waste time
		if @aria2["dir"] == nil
			if ENV["XDG_DOWNLOAD_DIR"] != nil
				@aria2["dir"] = ENV["XDG_DOWNLOAD_DIR"]
			else
				dir = nil
				begin
					File.open(ENV["XDG_CONFIG_HOME"]+"/user-dirs.dirs") do |f|
						f.grep(/^XDG_DOWNLOAD_DIR=/) do |line|
							dir = line.sub(/^XDG_DOWNLOAD_DIR="(.*)"\n$/, '\1').sub('$HOME', ENV['HOME'])
						end
					end
				rescue
				end

				if dir != nil
					@aria2["dir"] = dir
				elsif Dir.exists?(ENV["HOME"]+"/Downloads")
					@aria2["dir"] = ENV["HOME"]+"/Downloads"
				else
					@aria2["dir"] = ENV["HOME"]
				end
			end
		end

		# Default preferred locations
		if @aria2["metalink-location"].empty?
			@aria2["metalink-location"] = "fr,de,be,se" # TODO read from current locale
		end

		# XML-RPC Server
		@values["custom-server"] = false
		@values["xmlrpc-host"] = "127.0.0.1"
		@values["xmlrpc-port"] = 6800
		@values["enable-upnp"] = false
	end

	def parse()
		@values.clear
		default_values
		begin
			file = File.open(@conf_dir+"eatmonkeyrc")
			while !file.eof and line = file.readline
				next if line !~ /^([a-zA-Z0-9-]+)=(.*)$/
				key = $1
				val = $2.strip
				if val == "true"
					val = true
				elsif val == "false"
					val = false
				elsif val =~ /^[0-9]+$/
					val = val.to_i
				end
				@values[key] = val
			end
			file.close
		rescue
		end
	end

	public

	def [](key)
		@values[key]
	end

	def []=(key, val)
		@values[key] = val
	end

	def save()
		aria2.save
		begin
			file = File.open(@conf_dir+"eatmonkeyrc", "w")
			@values.each do |key,val|
				file.puts("#{key}=#{val}")
			end
			file.close
		rescue
			p $!
		end
	end

end



class Eat::SettingsDialog

	attr_reader :dialog

	def initialize(parent_window = nil)
		builder = Builder.new
		builder << "settings.ui"

		@dialog = builder["dialog-settings"]
		@notebook = builder["notebook-settings"]
		@xmlrpc = Hash.new
		@xmlrpc[:vbox] = builder["xmlrpc-vbox"]
		@xmlrpc[:custom_server] = builder["action-custom-server"]
		@xmlrpc[:host] = builder["xmlrpc-host"]
		@xmlrpc[:port] = builder["xmlrpc-port"]
		@xmlrpc[:user] = builder["xmlrpc-user"]
		@xmlrpc[:passwd] = builder["xmlrpc-passwd"]
		@basic_options = Hash.new
		@basic_options[:frame] = builder["basic-options-frame"]
		@basic_options[:max_concurrent_downloads] = builder["max-concurrent-downloads"]
		@basic_options[:max_download_speed] = builder["max-download-speed"]
		@basic_options[:download_dir_button] = builder["download-dir-button"]
		@bittorrent = Hash.new
		@bittorrent[:frame] = builder["bittorrent-frame"]
		@bittorrent[:max_upload_speed] = builder["max-upload-speed"]
		@bittorrent[:max_peers] = builder["max-peers"]
		@bittorrent[:seed_ratio] = builder["seed-ratio"]
		@bittorrent[:enable_dht] = builder["enable-dht"]
		@bittorrent[:enable_pex] = builder["enable-pex"]
		@bittorrent[:require_encryption] = builder["require-encryption"]
		@advanced = Hash.new
		@advanced[:vbox] = builder["advanced-vbox"]
		@advanced[:split] = builder["split"]
		@advanced[:locations] = builder["locations"]
		@advanced[:listen_port] = builder["listen-port"]
		@advanced[:enable_upnp] = builder["enable-upnp"]
		@advanced[:certificates] = builder["certificates"]
		@advanced[:save_ondisk] = builder["save-ondisk"]

		# Setup parent window
		@dialog.set_transient_for(parent_window)

		# Setup actions
		builder["action-custom-server"].signal_connect("toggled") { action_custom_server }

		# Setup basic configuration values
		@settings = Eat::Settings.instance
		@xmlrpc[:custom_server].active = @settings["custom-server"]
		@xmlrpc[:host].text = @settings["xmlrpc-host"] if @settings["xmlrpc-host"] != nil
		@xmlrpc[:port].value = @settings["xmlrpc-port"] if @settings["xmlrpc-port"] != nil
		@xmlrpc[:user].text = @settings["xmlrpc-user"] if @settings["xmlrpc-user"] != nil
		@xmlrpc[:passwd].text = @settings["xmlrpc-passwd"] if @settings["xmlrpc-passwd"] != nil
		@basic_options[:max_concurrent_downloads].value = @settings.aria2["max-concurrent-downloads"]
		@basic_options[:max_download_speed].value = @settings.aria2["max-overall-download-limit"]
		@basic_options[:download_dir_button].current_folder = @settings.aria2["dir"]
		@bittorrent[:max_upload_speed].value = @settings.aria2["max-overall-upload-limit"]
		@bittorrent[:max_peers].value = @settings.aria2["bt-max-peers"]
		@bittorrent[:seed_ratio].value = @settings.aria2["seed-ratio"]
		@bittorrent[:enable_dht].active = @settings.aria2["enable-dht"]
		@bittorrent[:enable_pex].active = @settings.aria2["enable-peer-exchange"]
		@bittorrent[:require_encryption].active = @settings.aria2["bt-require-crypto"]
		@advanced[:split].value = @settings.aria2["split"]
		@advanced[:locations].text = @settings.aria2["metalink-location"]
		@advanced[:listen_port].value = @settings.aria2["listen-port"].to_i
		@advanced[:enable_upnp].active = @settings["enable-upnp"]
		@advanced[:certificates].active = @settings.aria2["check-certificate"]
		@advanced[:save_ondisk].active = @settings.aria2["follow-torrent"] == true

		# Setup advanced configuration treeview
		@treestore = TreeStore.new(String, String, TrueClass)
		builder["treeview-configuration"].set_model(@treestore)
		fill_treestore()

		cell = CellRendererText.new
		cell.ellipsize = Pango::ELLIPSIZE_END
		column = builder["treeviewcolumn-option"]
		column.pack_start(cell, true)
		column.add_attribute(cell, :text, 0)

		cell = CellRendererText.new
		cell.ellipsize = Pango::ELLIPSIZE_END
		column = builder["treeviewcolumn-value"]
		column.pack_start(cell, true)
		column.add_attribute(cell, :text, 1)
		column.add_attribute(cell, :editable, 2)
		cell.signal_connect("edited") { |this, path, new_text| treestore_update_iter(path, new_text) }

		# Setup navigation between basic/advanced configuration editing
		builder["button-edit-configuration"].signal_connect("clicked") { switch_configuration(true) }
		builder["button-back"].signal_connect("clicked") { switch_configuration(false) }

		@dialog.vbox.show_all
	end

	def show()
		@dialog.show_all
	end

	def run()
		res = @dialog.run
		update_settings if res == Dialog::RESPONSE_OK

		begin
			yield res
		rescue
		end
		@dialog.hide

		res
	end

	private

	def action_custom_server()
		active = @xmlrpc[:custom_server].active?
		@xmlrpc[:vbox].set_sensitive(active)
		@basic_options[:frame].set_sensitive(!active)
		@bittorrent[:frame].set_sensitive(!active)
		@advanced[:vbox].set_sensitive(!active)
	end

	def switch_configuration(advanced)
		@notebook.set_page(advanced ? 1 : 0)

		if advanced
			# Sync treestore with widget values
			@treeiter["max-concurrent-downloads"].set_value(1, @basic_options[:max_concurrent_downloads].value_as_int.to_s)
			@treeiter["max-overall-download-limit"].set_value(1, @basic_options[:max_download_speed].value_as_int.to_s)
			@treeiter["dir"].set_value(1, @basic_options[:download_dir_button].current_folder)
			@treeiter["max-overall-upload-limit"].set_value(1, @bittorrent[:max_upload_speed].value_as_int.to_s)
			@treeiter["bt-max-peers"].set_value(1, @bittorrent[:max_peers].value_as_int.to_s)
			@treeiter["seed-ratio"].set_value(1, @bittorrent[:seed_ratio].value.to_s)
			@treeiter["enable-dht"].set_value(1, @bittorrent[:enable_dht].active?.to_s)
			@treeiter["enable-peer-exchange"].set_value(1, @bittorrent[:enable_pex].active?.to_s)
			@treeiter["bt-require-crypto"].set_value(1, @bittorrent[:require_encryption].active?.to_s)
			@treeiter["split"].set_value(1, @advanced[:split].value_as_int.to_s)
			@treeiter["metalink-location"].set_value(1, @advanced[:locations].text)
			@treeiter["listen-port"].set_value(1, @advanced[:listen_port].value_as_int.to_s)
			@treeiter["check-certificate"].set_value(1, @advanced[:certificates].active?.to_s)
			@treeiter["follow-torrent"].set_value(1, @advanced[:save_ondisk].active? ? "true" : "mem")
		else
			# Sync widget values with treestore
			@basic_options[:max_concurrent_downloads].value = @treeiter["max-concurrent-downloads"].get_value(1).to_i
			@basic_options[:max_download_speed].value = @treeiter["max-overall-download-limit"].get_value(1).to_i
			@basic_options[:download_dir_button].current_folder = @treeiter["dir"].get_value(1)
			@bittorrent[:max_upload_speed].value = @treeiter["max-overall-upload-limit"].get_value(1).to_i
			@bittorrent[:max_peers].value = @treeiter["bt-max-peers"].get_value(1).to_i
			@bittorrent[:seed_ratio].value = @treeiter["seed-ratio"].get_value(1).to_f
			@bittorrent[:enable_dht].active = @treeiter["enable-dht"].get_value(1) == "true"
			@bittorrent[:enable_pex].active = @treeiter["enable-peer-exchange"].get_value(1) == "true"
			@bittorrent[:require_encryption].active = @treeiter["bt-require-crypto"].get_value(1) == "true"
			@advanced[:split].value = @treeiter["split"].get_value(1).to_i
			@advanced[:locations].text = @treeiter["metalink-location"].get_value(1)
			@advanced[:listen_port].value = @treeiter["listen-port"].get_value(1).to_i
			@advanced[:certificates].active = @treeiter["check-certificate"].get_value(1) == "true"
			@advanced[:save_ondisk].active = @treeiter["follow-torrent"].get_value(1) == "true"
		end
	end

	def update_settings()
		@settings["custom-server"] = @xmlrpc[:custom_server].active?
		@settings["xmlrpc-host"] = @xmlrpc[:host].text
		@settings["xmlrpc-port"] = @xmlrpc[:port].value_as_int
		@settings["xmlrpc-user"] = @xmlrpc[:user].text
		@settings["xmlrpc-passwd"] = @xmlrpc[:passwd].text
		@settings["enable-upnp"] = @advanced[:enable_upnp].active?

		@treeiter.each do |key, iter|
			@settings.aria2[key] = iter.get_value(1)
		end

		if @notebook.page == 0
			@settings.aria2["max-concurrent-downloads"] = @basic_options[:max_concurrent_downloads].value_as_int
			@settings.aria2["max-overall-download-limit"] = @basic_options[:max_download_speed].value_as_int
			@settings.aria2["dir"] = @basic_options[:download_dir_button].current_folder
			@settings.aria2["max-overall-upload-limit"] = @bittorrent[:max_upload_speed].value_as_int
			@settings.aria2["bt-max-peers"] = @bittorrent[:max_peers].value_as_int
			@settings.aria2["seed-ratio"] = @bittorrent[:seed_ratio].value
			@settings.aria2["enable-dht"] = @bittorrent[:enable_dht].active?
			@settings.aria2["enable-peer-exchange"] = @bittorrent[:enable_pex].active?
			@settings.aria2["bt-require-crypto"] = @bittorrent[:require_encryption].active?
			@settings.aria2["split"] = @advanced[:split].value_as_int
			@settings.aria2["metalink-location"] = @advanced[:locations].text
			@settings.aria2["listen-port"] = @advanced[:listen_port].value_as_int
			@settings.aria2["check-certificate"] = @advanced[:certificates].active?
			@settings.aria2["follow-torrent"] = @advanced[:save_ondisk].active? ? "true" : "mem"
			@settings.aria2["follow-metalink"] = @advanced[:save_ondisk].active? ? "true" : "mem"
		end

		@settings.save
	end

	def treestore_append_iter(parent_iter, name)
		iter = @treestore.append(parent_iter)
		iter.set_value(0, name)
		iter.set_value(1, @settings.aria2[name].to_s)
		iter.set_value(2, true)
		@treeiter[name] = iter
	end

	def treestore_update_iter(path, new_name)
		iter = @treestore.get_iter(path)
		iter.set_value(1, new_name)
	end

	def fill_treestore()
		@treeiter = {}

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "Basic options")

		treestore_append_iter(parent_iter, "dir")
		treestore_append_iter(parent_iter, "max-concurrent-downloads")
		treestore_append_iter(parent_iter, "check-integrity")

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "HTTP/FTP options")

		treestore_append_iter(parent_iter, "all-proxy")
		treestore_append_iter(parent_iter, "connect-timeout")
		treestore_append_iter(parent_iter, "lowest-speed-limit")
		treestore_append_iter(parent_iter, "max-file-not-found")
		treestore_append_iter(parent_iter, "max-tries")
		treestore_append_iter(parent_iter, "proxy-method")
		treestore_append_iter(parent_iter, "remote-time")
		treestore_append_iter(parent_iter, "split")
		treestore_append_iter(parent_iter, "timeout")

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "HTTP options")

		treestore_append_iter(parent_iter, "ca-certificate")
		treestore_append_iter(parent_iter, "certificate")
		treestore_append_iter(parent_iter, "check-certificate")
		treestore_append_iter(parent_iter, "http-auth-challenge")
		treestore_append_iter(parent_iter, "http-user")
		treestore_append_iter(parent_iter, "http-passwd")
		treestore_append_iter(parent_iter, "http-proxy")
		treestore_append_iter(parent_iter, "https-proxy")
		treestore_append_iter(parent_iter, "private-key")
		treestore_append_iter(parent_iter, "referer")
		treestore_append_iter(parent_iter, "enable-http-keep-alive")
		treestore_append_iter(parent_iter, "enable-http-pipelining")
		treestore_append_iter(parent_iter, "header")
		treestore_append_iter(parent_iter, "load-cookies")
		treestore_append_iter(parent_iter, "save-cookies")
		treestore_append_iter(parent_iter, "use-head")
		treestore_append_iter(parent_iter, "user-agent")

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "FTP options")

		treestore_append_iter(parent_iter, "ftp-user")
		treestore_append_iter(parent_iter, "ftp-passwd")
		treestore_append_iter(parent_iter, "ftp-pasv")
		treestore_append_iter(parent_iter, "ftp-proxy")
		treestore_append_iter(parent_iter, "ftp-type")
		treestore_append_iter(parent_iter, "ftp-reuse-connection")

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "BitTorrent options")

		treestore_append_iter(parent_iter, "bt-external-ip")
		treestore_append_iter(parent_iter, "bt-hash-check-seed")
		treestore_append_iter(parent_iter, "bt-max-open-files")
		treestore_append_iter(parent_iter, "bt-max-peers")
		treestore_append_iter(parent_iter, "bt-min-crypto-level")
		treestore_append_iter(parent_iter, "bt-prioritize-piece")
		treestore_append_iter(parent_iter, "bt-require-crypto")
		treestore_append_iter(parent_iter, "bt-request-peer-speed-limit")
		treestore_append_iter(parent_iter, "bt-save-metadata")
		treestore_append_iter(parent_iter, "bt-seed-unverified")
		treestore_append_iter(parent_iter, "bt-stop-timeout")
		treestore_append_iter(parent_iter, "bt-tracker-interval")
		treestore_append_iter(parent_iter, "dht-entry-point")
		treestore_append_iter(parent_iter, "dht-listen-port")
		treestore_append_iter(parent_iter, "enable-dht")
		treestore_append_iter(parent_iter, "enable-peer-exchange")
		treestore_append_iter(parent_iter, "follow-torrent")
		treestore_append_iter(parent_iter, "listen-port")
		treestore_append_iter(parent_iter, "max-overall-upload-limit")
		treestore_append_iter(parent_iter, "max-upload-limit")
		treestore_append_iter(parent_iter, "peer-id-prefix")
		treestore_append_iter(parent_iter, "seed-ratio")

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "Metalink options")

		treestore_append_iter(parent_iter, "follow-metalink")
		treestore_append_iter(parent_iter, "metalink-servers")
		treestore_append_iter(parent_iter, "metalink-language")
		treestore_append_iter(parent_iter, "metalink-location")
		treestore_append_iter(parent_iter, "metalink-os")
		treestore_append_iter(parent_iter, "metalink-version")
		treestore_append_iter(parent_iter, "metalink-preferred-protocol")
		treestore_append_iter(parent_iter, "metalink-enable-unique-protocol")

		parent_iter = @treestore.append(nil)
		parent_iter.set_value(0, "Advanced options")

		treestore_append_iter(parent_iter, "allow-overwrite")
		treestore_append_iter(parent_iter, "allow-piece-length-change")
		treestore_append_iter(parent_iter, "async-dns")
		treestore_append_iter(parent_iter, "auto-file-renaming")
		treestore_append_iter(parent_iter, "auto-save-interval")
		treestore_append_iter(parent_iter, "enable-direct-io")
		treestore_append_iter(parent_iter, "file-allocation")
		treestore_append_iter(parent_iter, "interface")
		treestore_append_iter(parent_iter, "on-download-complete")
		treestore_append_iter(parent_iter, "on-download-error")
		treestore_append_iter(parent_iter, "on-download-start")
		treestore_append_iter(parent_iter, "on-download-stop")
		treestore_append_iter(parent_iter, "summary-interval")
		treestore_append_iter(parent_iter, "force-sequential")
		treestore_append_iter(parent_iter, "max-overall-download-limit")
		treestore_append_iter(parent_iter, "max-download-limit")
		treestore_append_iter(parent_iter, "no-file-allocation-limit")
		treestore_append_iter(parent_iter, "realtime-chunk-checksum")
	end

end



if __FILE__ == $0
	dialog = Eat::SettingsDialog.new
	dialog.run
end

