#!/usr/bin/ruby -w

require "gtk2"
require "singleton"

require "eataria2"

include Gtk

module Eat
end



class Eat::Settings

	include Singleton

	attr_reader :aria2
	attr_accessor :values

	def initialize()
		@conf_dir = ENV["XDG_CONFIG_HOME"]+"/eatmonkey/"
		@aria2 = Eat::Aria2Config.new(@conf_dir+"aria2.conf")
		@values = {}
		parse
	end

	private

	def default_values()
		# Download Directory
		# g_get_user_special_dir(G_USER_DIRECTORY_DOWNLOAD) or how not to waste time
		if @aria2["dir"] == nil
			if ENV["XDG_DOWNLOAD_DIR"] != nil
				@aria2["dir"] = ENV["XDG_DOWNLOAD_DIR"]
			else
				dir = nil
				begin
					File.open(ENV["XDG_CONFIG_HOME"]+"/user-dirs.dirs") do |f|
						f.grep(/^XDG_DOWNLOAD_DIR=/) do |line|
							dir = line.sub(/^XDG_DOWNLOAD_DIR="(.*)"\n$/, '\1').sub('$HOME', ENV['HOME'])
						end
					end
				rescue
				end

				if dir != nil
					@aria2["dir"] = dir
				elsif Dir.exists?(ENV["HOME"]+"/Downloads")
					@aria2["dir"] = ENV["HOME"]+"/Downloads"
				else
					@aria2["dir"] = ENV["HOME"]
				end
			end
		end

		# XML-RPC Server
		@values["custom-server"] = false
		@values["xmlrpc-host"] = "127.0.0.1"
		@values["xmlrpc-port"] = 6800
	end

	def parse()
		@values.clear
		default_values
		begin
			file = File.open(@conf_dir+"eatmonkeyrc")
			while !file.eof and line = file.readline
				next if line !~ /^([a-zA-Z0-9-]+)=(.*)$/
				key = $1
				val = $2.strip
				if val == "true"
					val = true
				elsif val == "false"
					val = false
				elsif val =~ /^[0-9]+$/
					val = val.to_i
				end
				@values[key] = val
			end
			file.close
		rescue
		end
	end

	public

	def [](key)
		@values[key]
	end

	def []=(key, val)
		@values[key] = val
	end

	def save()
		aria2.save
		begin
			file = File.open(@conf_dir+"eatmonkeyrc", "w")
			@values.each do |key,val|
				file.puts("#{key}=#{val}")
			end
			file.close
		rescue
			p $!
		end
	end

end



class Eat::SettingsDialog

	attr_reader :dialog

	def initialize(parent_window = nil)
		builder = Builder.new
		builder << "settings.ui"

		@dialog = builder["dialog-settings"]
		@xmlrpc = Hash.new
		@xmlrpc[:vbox] = builder["xmlrpc-vbox"]
		@xmlrpc[:custom_server] = builder["action-custom-server"]
		@xmlrpc[:host] = builder["xmlrpc-host"]
		@xmlrpc[:port] = builder["xmlrpc-port"]
		@xmlrpc[:user] = builder["xmlrpc-user"]
		@xmlrpc[:passwd] = builder["xmlrpc-passwd"]
		@basic_options = Hash.new
		@basic_options[:frame] = builder["basic-options-frame"]
		@basic_options[:max_concurrent_downloads] = builder["max-concurrent-downloads"]
		@basic_options[:max_download_speed] = builder["max-download-speed"]
		@basic_options[:download_dir_button] = builder["download-dir-button"]
		@bittorrent = Hash.new
		@bittorrent[:frame] = builder["bittorrent-frame"]
		@bittorrent[:max_upload_speed] = builder["max-upload-speed"]
		@bittorrent[:max_peers] = builder["max-peers"]
		@bittorrent[:seed_ratio] = builder["seed-ratio"]
		@bittorrent[:enable_dht] = builder["enable-dht"]
		@bittorrent[:enable_pex] = builder["enable-pex"]
		@bittorrent[:require_encryption] = builder["require-encryption"]

		# Setup parent window
		@dialog.set_transient_for(parent_window)

		# Setup actions
		builder["action-custom-server"].signal_connect("toggled") { action_custom_server }

		# Setup values
		@settings = Eat::Settings.instance
		@xmlrpc[:custom_server].active = @settings["custom-server"]
		@xmlrpc[:host].text = @settings["xmlrpc-host"] if @settings["xmlrpc-host"] != nil
		@xmlrpc[:port].value = @settings["xmlrpc-port"] if @settings["xmlrpc-port"] != nil
		@xmlrpc[:user].text = @settings["xmlrpc-user"] if @settings["xmlrpc-user"] != nil
		@xmlrpc[:passwd].text = @settings["xmlrpc-passwd"] if @settings["xmlrpc-passwd"] != nil
		@basic_options[:max_concurrent_downloads].value = @settings.aria2["max-concurrent-downloads"]
		@basic_options[:max_download_speed].value = @settings.aria2["max-overall-download-limit"]
		@basic_options[:download_dir_button].current_folder = @settings.aria2["dir"]
		@bittorrent[:max_upload_speed].value = @settings.aria2["max-overall-upload-limit"]
		@bittorrent[:max_peers].value = @settings.aria2["bt-max-peers"]
		@bittorrent[:seed_ratio].value = @settings.aria2["seed-ratio"]
		@bittorrent[:enable_dht].active = @settings.aria2["enable-dht"]
		@bittorrent[:enable_pex].active = @settings.aria2["enable-peer-exchange"]
		@bittorrent[:require_encryption].active = @settings.aria2["bt-require-crypto"]

		@dialog.vbox.show_all
	end

	def show()
		@dialog.show_all
	end

	def run()
		# Run dialog
		res = @dialog.run

		# Update settings
		if res == Dialog::RESPONSE_OK
			@settings["custom-server"] = @xmlrpc[:custom_server].active?
			@settings["xmlrpc-host"] = @xmlrpc[:host].text
			@settings["xmlrpc-port"] = @xmlrpc[:port].value_as_int
			@settings["xmlrpc-user"] = @xmlrpc[:user].text
			@settings["xmlrpc-passwd"] = @xmlrpc[:passwd].text
			@settings.aria2["max-concurrent-downloads"] = @basic_options[:max_concurrent_downloads].value_as_int
			@settings.aria2["max-overall-download-limit"] = @basic_options[:max_download_speed].value_as_int
			@settings.aria2["dir"] = @basic_options[:download_dir_button].current_folder
			@settings.aria2["max-overall-upload-limit"] = @bittorrent[:max_upload_speed].value_as_int
			@settings.aria2["bt-max-peers"] = @bittorrent[:max_peers].value_as_int
			@settings.aria2["seed-ratio"] = @bittorrent[:seed_ratio].value
			@settings.aria2["enable-dht"] = @bittorrent[:enable_dht].active?
			@settings.aria2["enable-peer-exchange"] = @bittorrent[:enable_pex].active?
			@settings.aria2["bt-require-crypto"] = @bittorrent[:require_encryption].active?
			@settings.save
		end

		begin
			yield res
		rescue
		end
		@dialog.hide

		res
	end

	private

	def action_custom_server()
		active = @xmlrpc[:custom_server].active?
		@xmlrpc[:vbox].set_sensitive(active)
		@basic_options[:frame].set_sensitive(!active)
		@bittorrent[:frame].set_sensitive(!active)
	end

end



if __FILE__ == $0
	dialog = Eat::SettingsDialog.new
	dialog.run
end

